<?php

namespace Modules\Contact\app\Http\Controllers;

use App\Exports\ContactExport;
use Illuminate\Http\Request;
use Illuminate\Http\Response;
use Maatwebsite\Excel\Facades\Excel;
use Modules\Contact\app\Models\ContactUs;
use Modules\Mwz\app\Http\Controllers\AdminController;

class ContactUsAdminController extends AdminController
{
	protected $config;

	/**
	 * Function : construct
	 * Dev : Poom
	 * Update Date : 29 Apr 2024
	 */
	public function __construct()
	{
		$this->middleware('auth:admin');

		self::$navbar = [
			['name' => __("contact::list.title"), 'url' => null],
			['name' => __("contact::list.name"), 'url' => route_has("admin.contact.list.index")]
		];

		$this->config = config('contact.list');
	}

	/**
	 * Function : contact list index
	 * Dev : Poom
	 * Update Date : 17 Apr 2024
	 * @return \Illuminate\Contracts\View\View contact::list.index
	 */
	public function index()
	{
		return view('contact::list.index', ['navbar' => self::$navbar, 'config' => $this->config]);
	}

	/**
	 * Function : contact list datatable ajax
	 * Dev : Poom
	 * Update Date : 17 Apr 2024
	 * @param \Illuminate\Http\Request $request
	 * @return Yajra\DataTables\Facades\DataTables data table
	 */
	public function datatable_ajax(Request $request)
	{
		if ($request->ajax()) {
			//init datatable

			$dt_column = ['id', 'name'];
			if ($this->config['company']) {
				array_push($dt_column, 'company');
			}
			if ($this->config['email'] || $this->config['tel']) {
				array_push($dt_column, 'contact');
			}
			if ($this->config['line']) {
				array_push($dt_column, 'line_id');
			}
			if ($this->config['subject']) {
				array_push($dt_column, 'subject');
			}
			array_push($dt_column, 'created_at', 'action');
			$dt_order = $request->get('order')[0]['column'];
			$dt_dir = $request->get('order')[0]['dir'];
			$dt_start = $request->get('start');
			$dt_length = $request->get('length');
			$dt_search = $request->get('search')['value'];
			$dt_filter = $request->get('filter');

			// create menu cat object
			$req = new ContactUs();

			// parent_id
			if (!empty($dt_filter['period']))
				$req =  self::mwz_filter($req, $dt_filter['period'], 'created_at', $dt_filter['range']);
			// dt_search 
			if (!empty($dt_search)) {
				$req = $req->where(function ($query) use ($dt_search) {
					$query->orWhere('name', 'like', "%$dt_search%");
				});
			}

			// count all menu cat
			$dt_total = $req->count();
			// set query order & limit from datatable
			$resp  = $req->orderBy($dt_column[$dt_order], $dt_dir)
				->offset($dt_start)
				->limit($dt_length)
				->get();

			// prepare datatable for response
			$tables = datatables($resp)
				->addIndexColumn()
				->setRowId('id')
				->setRowClass('contact_row')
				->setTotalRecords($dt_total)
				->setFilteredRecords($dt_total)
				->setOffset($dt_start)
				->editColumn('created_at', function ($record) {
					return str_replace(' ', '<br>', date("Y-m-d H:i:s", strtotime($record->created_at)));
				})->editColumn('name', function ($record) {
					return str_implode(' ', $record->name ?? '', $record->surname ?? '');
				})->editColumn('company', function ($record) {
					return limit($record->company ?? '', 50);
				})->editColumn('contact', function ($record) {
					return str_implode(
						'<br>',
						!empty($record->email) ? __('field.email') . ' : ' . $record->email : null,
						!empty($record->email) ? __('field.tel') . ' : ' . ($record->tel ?? '') : null
					);
				})->editColumn('line_id', function ($record) {
					return limit($record->line_id ?? '', 50);
				})->editColumn('subject', function ($record) {
					return limit($record->subject ?? '', 50);
				})->addColumn('action', function ($record) {
					$btn = '<div class="button-list">';
					if (!empty($this->config['btn']['edit'])) {
						$btn .= self::btn_route("admin.contact.list.view", $record->id, 'btn-white', 'fe-menu');
					}
					if (!empty($this->config['btn']['delete'])) {
						$btn .= self::btn_delete("admin.contact.list.set_delete", $record->id, 'setDelete');
					}
					$btn .= '</div>';
					return $btn;
				})->escapeColumns([]);

			// response datatable json
			return $tables->make(true);
		}
	}

	/**
	 * Function : contact list form
	 * Dev : Poom
	 * Update Date : 17 Apr 2024
	 * @param int $id
	 * @return \Illuminate\Contracts\View\View contact::list.form
	 */
	public function  form($id)
	{
		$data = ContactUs::find($id);

		return view('contact::list.form', ['data' => $this->arr_contact($data), 'navbar' => self::$navbar]);
	}

	/**
	 * Function : contact list array data
	 * Dev : Poom
	 * Update Date : 17 Apr 2024
	 * @param mixed $param
	 * @return array data
	 */
	private function arr_contact($param)
	{
		$data = [
			['name' => __('field.name'), 'value' => str_implode(' ', $param->name, $param->surname ?? '')],
			['name' => __('field.email'), 'value' => $param->email],
			['name' => __('field.tel'), 'value' => $param->tel],
			['name' => __('field.subject'), 'value' => $param->subject ?? ''],
			['name' => __('field.detail'), 'value' => $param->msg],
			// ['name' => __('field.created'), 'value' => str_date($param->created_at, 'd m Y H:i')]
		];
		return $data;
	}

	/**
	 * Function : contact list set delete
	 * Dev : Poom
	 * Update Date : 17 Apr 2024
	 * @param \Illuminate\Http\Request $request
	 * @return \Illuminate\Http\JsonResponse
	 */
	public function set_delete(Request $request)
	{
		$set = ContactUs::find($request->get('id'));
		if ($set->delete())
			return self::response(['msg' => __('noti.delete')]);

		return self::response(['msg' => __('noti.error')], Response::HTTP_MOVED_PERMANENTLY);
	}

	/**
	 * Function : contact list export
	 * Dev : Poom
	 * Update Date : 17 Apr 2024
	 * @param \Illuminate\Http\Request $request
	 * @return \Illuminate\Http\RedirectResponse
	 */
	public function export(Request $request)
	{
		$time = time();
		$request->session()->forget('errors');
		$dt_filter = $request->get('filter');
		$req = new ContactUs();
		if (!empty($dt_filter['period']))
			$req = self::mwz_filter($req, $dt_filter['period'], 'created_at', $dt_filter['range']);

		$row = $req->count();
		$data = $req->orderBy('created_at', 'desc')->get();
		if (!empty($data) && $row > 0)
			return Excel::download(new ContactExport($data), "contact-{$time}.xlsx");

		$request->session()->regenerate();
		$request->session()->put('errors', __('noti.no-data'));

		return redirect()->back();
	}
}
