<?php

namespace Modules\Content\app\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Http\Response;
use Illuminate\Support\Facades\Route;
use Modules\Content\app\Models\ContentCategories;
use Modules\Mwz\app\Http\Controllers\AdminController;
use Modules\Mwz\app\Http\Controllers\SlugController;

class ContentCategoryAdminController extends AdminController
{
	private $module, $method, $action, $config;

	/**
	 * Function : construct
	 * Dev : Petch
	 * Update Date : 23 Jan 2025
	 */
	public function __construct()
	{
		$routeName = Route::currentRouteName();
		if ($routeName) {
			$explode = explode('.', $routeName);
			$this->module = $explode[1] ?? '';
			$this->method = $explode[2] ?? '';
			$this->action = $explode[3] ?? '';
		} else {
			$this->module = '';
			$this->method = '';
			$this->action = '';
		}

		self::$navbar = [
			['name' => __("content::category.{$this->module}.title"), 'url' => null],
			['name' => __("content::category.{$this->module}.name"), 'url' => route_has("admin.{$this->module}.{$this->method}.index")]
		];

		$this->config = config("content.category.{$this->module}");
	}

	/**
	 * Function : construct
	 * Dev : Petch
	 * Update Date : 23 Jan 2025
	 */
	public function index()
	{
		$limit = false;
		if (!empty($this->config['setting']['limit']['status']))
			$limit = ContentCategories::where('type', $this->module)->whereNotNull('parent_id')->count() >= $this->config['setting']['limit']['max'];
		return view('content::category.index', [
			'navbar' => self::$navbar,
			'method' => $this->method,
			'type' => $this->module,
			'config' => $this->config,
			'limit' => $limit
		]);
	}

	/**
	 * Function : construct
	 * Dev : Petch
	 * Update Date : 23 Jan 2025
	 */
	public function datatable_ajax(Request $request)
	{
		if ($request->ajax()) {
			//init datatable
			$dt_column = ['_lft'];
			foreach ($this->config['table']['header'] as $h) {
				if (!empty($h['status']))
					$dt_column[] = $h['column'];
			}
			if (!empty($this->config['table']['btn']['sort']))
				$dt_column[] = 'sort';
			array_push($dt_column, 'updated_at', 'action');

			$dt_order = $request->get('order')[0]['column'];
			$dt_dir = $request->get('order')[0]['dir'];
			$dt_start = $request->get('start');
			$dt_length = $request->get('length');
			$dt_search = $request->get('search')['value'];
			$dt_filter = $request->get('filter');

			// create Category object
			$req = ContentCategories::where('type', $this->module)
				->whereNotNull('parent_id');

			// dt_search 
			if (!empty($dt_search))
				$req = $req->whereRelation('local', 'name', 'like', "%{$dt_search}%");

			// dt_filter
			if (!empty($dt_filter['parent_id']))
				$req = $req->whereDescendantOrSelf($dt_filter['parent_id']);

			// count all 
			$dt_total = $req->count();

			// set query order & limit from datatable
			$resp = $req->orderBy($dt_column[$dt_order], $dt_dir)
				->offset($dt_start)
				->limit($dt_length)
				->withDepth()
				->defaultOrder()
				->get();

			// prepare datatable for response
			$tables = datatables($resp)
				->addIndexColumn()
				->setRowId('id')
				->setRowClass('content_row')
				->setTotalRecords($dt_total)
				->setFilteredRecords($dt_total)
				->setOffset($dt_start)
				->editColumn('image', function ($record) {
					return $this->view_file('image', $record->image);
				})->editColumn('name', function ($record) {
					return str_repeat(' - ', $record->depth - 1) . ($record->local->name ?? '');
				})->editColumn('parent_id', function ($record) {
					return self::make_name($record->parent, $record->parent->local->name ?? '');
				})->editColumn('sort', function ($record) use ($dt_order) {
					if (!empty($this->config['table']['btn']['sort']))
						return self::btn_sort("admin.{$this->module}.{$this->method}.set_sort", $record->id, 'setSort', 0, $dt_order != 0);
				})->editColumn('updated_at', function ($record) {
					return str_replace(' ', '<br>', date("Y-m-d H:i:s", strtotime($record->updated_at)));
				})->addColumn('action', function ($record) {
					$btn = '<div class="button-list">';

					if (!empty($this->config['table']['btn']['default']))
						$btn .= self::btn_default("admin.{$this->module}.{$this->method}.set_default", $record->id, 'setDefault', $record->default);
					if (!empty($this->config['table']['btn']['status']))
						$btn .= self::btn_status("admin.{$this->module}.{$this->method}.set_status", $record->id, 'setStatus', $record->status);
					if (!empty($this->config['table']['btn']['edit']))
						$btn .= self::btn_route("admin.{$this->module}.{$this->method}.edit", $record->id);
					if (!empty($this->config['table']['btn']['delete']))
						$btn .= self::btn_delete("admin.{$this->module}.{$this->method}.set_delete", $record->id, 'setDelete');

					$btn .= '</div>';

					return $btn;
				})->escapeColumns([]);
			// response datatable json
			return $tables->make(true);
		}
	}

	/**
	 * Function : construct
	 * Dev : Petch
	 * Update Date : 31 Jan 2025
	 */
	public static function make_name($category, $name)
	{
		if (empty($category->parent) || empty($category->parent->local->name))
			return $name;

		return self::make_name($category->parent, "{$category->parent->local->name} &raquo; {$name}");
	}

	/**
	 * Function : construct
	 * Dev : Petch
	 * Update Date : 23 Jan 2025
	 */
	public function form($id = 0)
	{
		self::$navbar[] =  ['name' =>  __("action.{$this->action}"), 'url' => null];
		$meta = self::set_metadata($this->module,  "{$this->module}_{$this->method}", 3);
		$data = null;
		$limit = false;

		if (!empty($id)) {
			$data = ContentCategories::where([['id', $id], ['type', $this->module]])->first();
			if (empty($data))
				return abort(Response::HTTP_NOT_FOUND);

			if (!empty($data->langs)) {
				$data->langs = $data->langs->groupBy('lang')->map(function ($row) {
					return  $row->first();
				});
			}
			if (!empty($this->config['form']['navbar']['seo']['status']))
				$meta = array_merge($meta, SlugController::getMetadata($this->module, "{$this->module}_{$this->method}", $id));
		} elseif (!empty($this->config['setting']['limit']['status']))
			$limit = ContentCategories::where('type', $this->module)->whereNotNull('parent_id')->count() >= $this->config['setting']['limit']['max'];

		return view('content::category.form', [
			'data' => $data,
			'navbar' => self::$navbar,
			'type' => $this->module,
			'method' => $this->method,
			'meta' => $meta,
			'config' => $this->config,
			'limit' => $limit,
			'action' => $this->action
		]);
	}

	/**
	 * Function : construct
	 * Dev : Petch
	 * Update Date : 23 Jan 2025
	 */
	public function save(Request $request)
	{
		$rule = $msg = [];
		$navbar = $this->config['form']['navbar'];
		$sidebar = $this->config['form']['sidebar'];

		if (!empty($navbar['translate']['status'])) {
			foreach (languages() as $lang => $l) {
				foreach ($navbar['translate']['input'] as $k => $i) {
					if (!empty($i['status']) && !empty($i['validate'])) {
						$rule["{$k}_{$lang}"] = ($k != 'name') ? 'required' : 'required|max:255';
						$msg["{$k}_{$lang}.*"] = __("field.{$k}_placeholder") . ' ' . ($l['name'] ?? '');
					}
				}
			}
		}
		foreach ($sidebar as $s) {
			if (empty($s['status']) || empty($s['input']))
				continue;

			foreach ($s['input'] as $k => $i) {
				if (!empty($i['status']) && !empty($i['validate'])) {
					$rule["{$k}"] = 'required';
					$msg["{$k}.*"] = __("field.{$k}_placeholder");
				}
			}
		}

		$valid = validator($request->all(), $rule, $msg);
		if ($valid->fails())
			return self::response(['msg' => $valid->errors()->first(), 'error' => $valid->errors()], Response::HTTP_BAD_REQUEST);

		$attr = [];
		foreach ($sidebar as $s) {
			if (empty($s['status']) || empty($s['input']))
				continue;

			foreach ($s['input'] as $k => $i) {
				if (empty($i['status']))
					continue;

				if ($i['type'] == 'time') {
					$attr[$i['start']] = $request->get($i['start']);
					$attr[$i['end']] = $request->get($i['end']);
				} elseif ($i['type'] == 'icon') {
					if (!empty($i['start']) && !empty($i['end'])) {
						$exp = explode('to', str_replace(' ', '', $request->get($k)) ?? '');
						$attr[$i['start']] = !empty($exp[0]) ? date('Y-m-d H:i:s', strtotime($exp[0])) : null;
						$attr[$i['end']] = !empty($exp[1]) ? date('Y-m-d H:i:s', strtotime($exp[1])) : null;
					} else
						$attr[$k] = $request->get($k) ?? now();
				} else
					$attr[$k] = $k == 'status' ? ($request->get($k) ?? 0) : $request->get($k);
			}
		}

		if (!empty($navbar['upload']['status'])) {
			foreach ($navbar['upload']['input'] as $k_u => $u) {
				if (empty($u['status']))
					continue;

				if (str_contains($u['type'], 'upload_')) {
					$chk = $this->save_upload($request, $k_u, str_replace('upload_', '', $u['type']));
					if (!$chk['status'])
						return self::response(['msg' => ($s_b['label'] ?? '') . ' ' . $chk['msg']], Response::HTTP_NOT_ACCEPTABLE);
					$attr[$k_u] = $chk['result'];
				} else
					$attr[$k_u] = $request->get($k_u);
			}
		}
		if (empty($request->get('id'))) {
			$chk_level_0 = ContentCategories::where('type', $this->module)->whereNull('parent_id')->first();
			if (empty($chk_level_0)) {
				$re = ContentCategories::create(['type' => $this->module]);
				if (empty($re))
					return self::response(['msg' => __('noti.error')], Response::HTTP_INTERNAL_SERVER_ERROR);

				$attr['parent_id'] = $re->id ?? null;
			} else
				$attr['parent_id'] = $chk_level_0->id ?? null;
		} elseif (!empty($request->get('parent_id')))
			$attr['parent_id'] = $request->get('parent_id');
		else
			$attr['parent_id'] = ContentCategories::where('type', $this->module)->whereNull('parent_id')->first()->id ?? null;

		$set = ContentCategories::updateOrCreate(['id' => $request->get('id') ?? 0, 'type' => $this->module], $attr);
		if ($set->save()) {
			if (!empty($navbar['translate']['status']))
				self::setLangs($request, $set, $navbar['translate']['input']);

			if (!empty($navbar['seo']['status']))
				SlugController::createMetadata($request, $set->id);

			return self::response(['msg' => __('noti.success'), 'id' => $set->id,  'url' => route("admin.{$this->module}.{$this->method}.index")]);
		}

		return self::response(['msg' => __('noti.error')], Response::HTTP_INTERNAL_SERVER_ERROR);
	}

	/**
	 * Function : set translate
	 * Dev : Petch
	 * Update Date : 22 Jan 2025
	 */
	private function save_upload($request, $name, $type)
	{
		if (check_file_upload($request, $name, $type))
			return ['status' => false, 'msg' => __('noti.mime_image')];

		return ['status' => true, 'result' => self::set_image_upload($request, $name, "public/{$this->module}", "{$type}-")];
	}

	/**
	 * Function : set translate
	 * Dev : Petch
	 * Update Date : 23 Jan 2025
	 */
	private function setLangs(Request $request, $row, $config)
	{
		foreach (array_keys(languages()) as $lang) {
			$attr = [];
			foreach ($config as $key => $item)
				$attr[$key] = !empty($request->get("{$key}_{$lang}")) ? mwz_setTextString($request->get("{$key}_{$lang}"), $item['type'] == 'texteditor') : null;

			$row->langs()->updateOrCreate(['lang' => $lang], $attr);
		}
	}

	/**
	 * Function : set default
	 * Dev : Petch
	 * Update Date : 23 Jan 2025
	 */
	public function set_default(Request $request)
	{
		$limit = $this->config['setting']['limit_default'];
		$set = ContentCategories::find($request->get('id'));
		$set->default = !$set->default;

		if (!empty($set->default) && !empty($limit['status']) && (ContentCategories::where([['type', $this->module], ['default', 1]])->count() >= $limit['max']))
			return self::response(['msg' => __('noti.default.max', ['max' => $limit['max']])], Response::HTTP_BAD_REQUEST);
		elseif ($set->save())
			return self::response(['msg' => __('noti.success')]);

		return self::response(['msg' => __('noti.error')], Response::HTTP_INTERNAL_SERVER_ERROR);
	}

	/**
	 * Function : set status
	 * Dev : Petch
	 * Update Date : 23 Jan 2025
	 */
	public function set_status(Request $request)
	{
		$set = ContentCategories::find($request->get('id'));
		$set->status = !$set->status;

		if (!empty($this->config['form']['navbar']['seo']['status']))
			SlugController::slug_status($this->module, "{$this->module}_{$this->method}", $set->id, $set->status);

		if ($set->save())
			return self::response(['msg' => __('noti.success')]);

		return self::response(['msg' => __('noti.error')], Response::HTTP_INTERNAL_SERVER_ERROR);
	}

	/**
	 * Function : delete
	 * Dev : Petch
	 * Update Date : 23 Jan 2025
	 */
	public function set_delete(Request $request)
	{
		$set = ContentCategories::find($request->get('id'));
		$path_image = $set->image;
		if ($set->contents()->exists())
			$set->contents()->update(['status' => 0]);

		if ($set->delete()) {
			self::delete_image($path_image);
			self::re_order();

			if (!empty($this->config['form']['navbar']['seo']['status']))
				SlugController::remove_metadata($this->module,  "{$this->module}_{$this->method}", $request->get('id'));

			$limit = 0;
			if (!empty($this->config['setting']['limit']['status']))
				$limit = ContentCategories::where('type', $this->module)->whereNotNull('parent_id')->count() < $this->config['setting']['limit']['max'];

			return self::response(['msg' => __('noti.delete'), 'limit' => $limit]);
		}

		return self::response(['msg' => __('noti.error')], Response::HTTP_BAD_REQUEST);
	}

	/**
	 * Function : set sort
	 * Dev : Petch
	 * Update Date : 23 Jan 2025
	 */
	public function set_sort(Request $request)
	{
		$node = ContentCategories::find($request->get('id'));
		$move = $request->get('move') == 'up' ? $node->up() : $node->down();

		if ($move)
			return self::response(['msg' =>  __('noti.sort')]);

		return self::response(['msg' => __('noti.error')], Response::HTTP_BAD_REQUEST);
	}

	/**
	 * Function : re order
	 * Dev : Petch
	 * Update Date : 23 Jan 2025
	 */
	public static function re_order()
	{
		ContentCategories::fixTree();
	}

	/**
	 * Function : get list
	 * Dev : petch
	 * Update Date : 23 Jan 2025
	 */
	public function get_list(Request $request)
	{
		$filter = $request->get('parent');
		$req = ContentCategories::where('type', $this->module)
			->whereNotNull('parent_id');

		if (!empty($filter)) {
			if (!empty($filter['search']))
				$req = $req->whereRelation('local', 'name', 'like', "%{$filter['search']}%");
			if (!empty($filter['id']))
				$req = $req->where('id', '<>', $filter['id']);
		}

		if (!empty($this->config['setting']['limit_parent']['status']))
			$req = $req->having('depth', '<=', $this->config['setting']['limit_parent']['level']);

		$data = $req->withDepth()
			->defaultOrder()
			->get();

		$result = [];
		foreach ($data as $list)
			$result[] = ['id' => $list->id, 'text' => str_repeat(' - ', $list->depth - 1) . $list->local->name, 'image' => $list->image ?? ''];

		return self::response(['results' => $result], 200);
	}
}
