<?php

namespace Modules\Content\app\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Http\Response;
use Illuminate\Support\Facades\Route;
use Illuminate\Support\Facades\Storage;
use Modules\Content\app\Models\ContentCategories;
use Modules\Content\app\Models\Contents;
use Modules\Mwz\app\Http\Controllers\AdminController;
use Modules\Mwz\app\Http\Controllers\SlugController;

class ContentsAdminController extends AdminController
{
	private $module, $method, $action, $config;

	/**
	 * Function : construct
	 * Dev : Petch
	 * Update Date : 22 Jan 2025
	 */
	public function __construct()
	{
		$routeName = Route::currentRouteName();
		if ($routeName) {
			$explode = explode('.', $routeName);
			$this->module = $explode[1] ?? '';
			$this->method = $explode[2] ?? '';
			$this->action = $explode[3] ?? '';
		} else {
			$this->module = '';
			$this->method = '';
			$this->action = '';
		}

		self::$navbar = [
			['name' => __("content::content.{$this->module}.title"), 'url' => null],
			['name' => __("content::content.{$this->module}.name"), 'url' => route_has("admin.{$this->module}.{$this->method}.index")]
		];

		$this->config = config("content.content.{$this->module}");
	}

	/**
	 * Function : index
	 * Dev : Petch
	 * Update Date : 22 Jan 2025
	 */
	public function index()
	{
		$limit = false;
		if (!empty($this->config['setting']['limit']['status']))
			$limit = Contents::where('type', $this->module)->whereNotNull('parent_id')->count() >= $this->config['setting']['limit']['max'];

		return view('content::content.index', [
			'navbar' => self::$navbar,
			'method' => $this->method,
			'type' => $this->module,
			'config' => $this->config,
			'limit' => $limit
		]);
	}

	/**
	 * Function : datatable ajax
	 * Dev : Petch
	 * Update Date : 22 Jan 2025
	 */
	public function datatable_ajax(Request $request)
	{
		if ($request->ajax()) {
			//init datatable
			$dt_column = ['_lft'];
			foreach ($this->config['table']['header'] as $h) {
				if (!empty($h['status']))
					$dt_column[] = $h['column'];
			}
			if (!empty($this->config['table']['btn']['sort']))
				$dt_column[] = 'sort';
			array_push($dt_column, 'updated_at', 'action');

			$dt_order = $request->get('order')[0]['column'];
			$dt_dir = $request->get('order')[0]['dir'];
			$dt_start = $request->get('start');
			$dt_length = $request->get('length');
			$dt_search = $request->get('search')['value'];
			$dt_filter = $request->get('filter');

			// create Content object
			$req = Contents::where('type', $this->module)
				->whereNotNull('parent_id');

			// filter
			if (!empty($dt_filter['category_id']))
				$req = $req->whereRelation('categories', 'category_id', $dt_filter['category_id']);

			// search 
			if (!empty($dt_search))
				$req = $req->whereRelation('local', 'name', 'like', "%{$dt_search}%");

			// count all 
			$dt_total = $req->count();
			if ($dt_order == 0 && empty($this->config['table']['btn']['sort']) && !empty($this->config['form']['sidebar']['setting']['input']['publish_at']['status']))
				$req = $req->orderBy('publish_at', 'DESC');
			else
				$req = $req->orderBy($dt_column[$dt_order], $dt_dir);

			$resp =	$req->offset($dt_start)
				->limit($dt_length)
				->get();

			// prepare datatable for response
			$tables = datatables($resp)
				->addIndexColumn()
				->setRowId('id')
				->setRowClass('content_row')
				->setTotalRecords($dt_total)
				->setFilteredRecords($dt_total)
				->setOffset($dt_start)
				->editColumn('image', function ($record) {
					foreach ($this->config['form']['navbar']['upload']['input'] as $k => $v) {
						if (!empty($v['status']) && !empty($record->{$k})) {
							$type = $k == 'gallery' ? 'image' : $k;
							if ($k == 'youtube')
								$src = $record->image ?? null;
							elseif ($k == 'gallery')
								$src = json_decode($record->{$k} ?? null, 1)[0]['image'] ?? null;
							else
								$src =  $record->{$k} ?? null;

							return self::view_file($type, $src);
						}
					}
				})->editColumn('name', function ($record) {
					return limit($record->local->name ?? '', 50);
				})->editColumn('category', function ($record) {
					return self::badge($record->categories ?? '');
				})->editColumn('prefix', function ($record) {
					return $record->local->prefix ?? '';
				})->editColumn('period', function ($record) {
					return date_range(' - ', $record->start_at ?? null, $record->end_at ?? null);
				})->editColumn('time', function ($record) {
					return str_implode(
						' - ',
						!empty($record->start_time) ? date('H:i', strtotime($record->start_time)) : null,
						!empty($record->end_time) ? date('H:i', strtotime($record->end_time)) : null
					);
				})->editColumn('publish_at', function ($record) {
					return !empty($record->publish_at) ? str_replace(' ', '<br>', date("Y-m-d H:i:s", strtotime($record->publish_at))) : '-';
				})->editColumn('view', function ($record) {
					return number_format($record->view ?? 0);
				})->editColumn('sort', function ($record) use ($dt_order) {
					if (!empty($this->config['table']['btn']['sort']))
						return self::btn_sort("admin.{$this->module}.{$this->method}.set_sort", $record->id, 'setSort', 0, $dt_order != 0);
				})->editColumn('updated_at', function ($record) {
					return str_replace(' ', '<br>', date("Y-m-d H:i:s", strtotime($record->updated_at)));
				})->addColumn('action', function ($record) {
					$btn = '<div class="button-list">';

					if (!empty($this->config['table']['btn']['default']))
						$btn .= self::btn_default("admin.{$this->module}.{$this->method}.set_default", $record->id, 'setDefault', $record->default);
					if (!empty($this->config['table']['btn']['status']))
						$btn .= self::btn_status("admin.{$this->module}.{$this->method}.set_status", $record->id, 'setStatus', $record->status);
					if (!empty($this->config['table']['btn']['edit']))
						$btn .= self::btn_route("admin.{$this->module}.{$this->method}.edit", $record->id);
					if (!empty($this->config['table']['btn']['delete']))
						$btn .= self::btn_delete("admin.{$this->module}.{$this->method}.set_delete", $record->id, 'setDelete');

					$btn .= '</div>';

					return $btn;
				})->escapeColumns([]);
			// response datatable json
			return $tables->make(true);
		}
	}

	/**
	 * Function : form
	 * Dev : Petch
	 * Update Date : 22 Jan 2025
	 */
	public function form($id = 0)
	{
		self::$navbar[] =  ['name' => __("action.{$this->action}"), 'url' => null];
		$meta = self::set_metadata($this->module,  "{$this->module}_{$this->method}", 3);
		$data = null;
		$limit = false;
		if (!empty($id)) {
			$data = Contents::where([['id', $id], ['type', $this->module]])->first();
			if (empty($data))
				return abort(Response::HTTP_NOT_FOUND);

			if (!empty($data->langs)) {
				$data->langs = $data->langs->groupBy('lang')->map(function ($row) {
					return  $row->first();
				});
			}

			if (!empty($this->config['form']['navbar']['seo']['status']))
				$meta = array_merge($meta, SlugController::getMetadata($this->module, "{$this->module}_{$this->method}", $id));
		} elseif (!empty($this->config['setting']['limit']['status']))
			$limit = Contents::where('type', $this->module)->whereNotNull('parent_id')->count() >= $this->config['setting']['limit']['max'];

		return view('content::content.form', [
			'data' => $data,
			'navbar' => self::$navbar,
			'type' => $this->module,
			'method' => $this->method,
			'meta' => $meta,
			'config' => $this->config,
			'limit' => $limit,
			'action' => $this->action
		]);
	}

	public function save(Request $request)
	{
		$rule = $msg = [];
		$navbar = $this->config['form']['navbar'];
		$sidebar = $this->config['form']['sidebar'];
		$limit = $this->config['setting']['limit_default'] ?? null;

		if (!empty($limit['status']) && !empty($request->get('default')) && (Contents::where([['type', $this->module], ['default', 1], ['id', '!=', $request->get('id')]])->count() >= $limit['max']))
			return self::response(['msg' => __('noti.default.max', ['max' => $limit['max']])], Response::HTTP_BAD_REQUEST);

		if (!empty($navbar['translate']['status'])) {
			foreach (languages() as $lang => $l) {
				foreach ($navbar['translate']['input'] as $k => $i) {
					if (!empty($i['status']) && !empty($i['validate'])) {
						$rule["{$k}_{$lang}"] = ($k != 'name') ? 'required' : 'required|max:255';
						$msg["{$k}_{$lang}.*"] = __("field.{$k}_placeholder") . ' ' . ($l['name'] ?? '');
					}
				}
			}
		}
		foreach ($navbar as $k_n => $n) {
			if ($k_n == 'translate' || empty($n['status']) || empty($n['input']))
				continue;

			foreach ($n['input'] as $k => $i) {
				if (empty($i['status']) || empty($i['validate']))
					continue;

				if (in_array($k_n, ['upload', 'file', 'video'])) {
					if (empty($request->get('id')) || !empty($request->get($k . '_del')) || (empty($request->get($k . '_old') && empty($request->get($k)))) and !empty($i['validate']))
						$rule[$k] = 'required';
					else
						$rule[$k . '_old'] = ['required'];
				} else
					$rule[$k] = 'required';

				$msg["{$k}*.*"] = __("field.error.{$k}_placeholder");
			}
		}
		foreach ($sidebar as $s) {
			if (empty($s['status']) || empty($s['input']))
				continue;

			foreach ($s['input'] as $k => $i) {
				if (!empty($i['status']) && !empty($i['validate'])) {
					if ($i['type'] == 'time') {
						$rule["{$i['start']}"] = 'required';
						$rule["{$i['end']}"] = 'required';
						$msg["{$i['start']}.*"] = __("field.{$k}_placeholder");
						$msg["{$i['end']}.*"] = __("field.{$k}_placeholder");
					} else {
						$rule["{$k}"] = 'required';
						$msg["{$k}.*"] = __("field.{$k}_placeholder");
					}
				}
			}
		}

		// Check if link starts with http:// or https://
		if ($request->has('link') && !empty($request->get('link'))) {
			if (!preg_match('/^https?:\/\//', $request->get('link'))) {
				return self::response(['msg' => 'ลิงก์ต้องขึ้นต้นด้วย http:// หรือ https://'], Response::HTTP_BAD_REQUEST);
			}
		}

		$valid = validator($request->all(), $rule, $msg);
		if ($valid->fails())
			return self::response(['msg' => $valid->errors()->first(), 'error' => $valid->errors()], Response::HTTP_BAD_REQUEST);

		$attr = [];
		foreach ($sidebar as $s) {
			if (empty($s['status']) || empty($s['input']))
				continue;

			foreach ($s['input'] as $k => $i) {
				if (empty($i['status']) || $k == 'category_id')
					continue;

				elseif ($i['type'] == 'time') {
					$attr[$i['start']] = $request->get($i['start']);
					$attr[$i['end']] = $request->get($i['end']);
				} elseif ($i['type'] == 'icon') {
					if (!empty($i['start']) && !empty($i['end'])) {
						$exp = explode('to', str_replace(' ', '', $request->get($k)) ?? '');
						$attr[$i['start']] = !empty($exp[0]) ? date('Y-m-d H:i:s', strtotime($exp[0])) : null;
						$attr[$i['end']] = !empty($exp[1]) ? date('Y-m-d H:i:s', strtotime($exp[1])) : null;
					} else
						$attr[$k] = $request->get($k) ?? now();
				} else
					$attr[$k] = $k == 'status' ? ($request->get($k) ?? 0) : $request->get($k);
			}
		}
		if (!empty($navbar['upload']['status'])) {
			foreach ($navbar['upload']['input'] as $k_u => $u) {
				if ($u['type'] == 'gallery' || empty($u['status']))
					continue;
				if ($k_u == 'youtube' && !empty($request->get('youtube')))
					$attr['youtube'] = self::embed($request->get("youtube"));
				elseif (str_contains($u['type'], 'upload_')) {
					$chk = $this->save_upload($request, $k_u, str_replace('upload_', '', $u['type']));
					if (!$chk['status'])
						return self::response(['msg' => ($s_b['label'] ?? '') . ' ' . $chk['msg']], Response::HTTP_NOT_ACCEPTABLE);
					$attr[$k_u] = $chk['result'];
				} else
					$attr[$k_u] = $request->get($k_u);
			}
		}
		//style
		if (!empty($navbar['style']['status'])) {
			foreach ($navbar['style']['input'] as $k_s => $s) {
				$attr[$k_s] = null;

				if ($request->get('style') == 1 && ($k_s == 'link' || $k_s == 'youtube')) {
					$attr[$k_s] = $request->get($k_s);
				}

				if (!empty($s['type']) && $request->get('style') == 2 && str_contains($s['type'], 'upload_')) {
					$chk = $this->save_upload($request, $k_s, str_replace('upload_', '', $s['type']));
					if (!$chk['status']) {
						return self::response(['msg' => ($s['label'] ?? '') . ' ' . $chk['msg']], Response::HTTP_NOT_ACCEPTABLE);
					}
					$attr[$k_s] = $chk['result'];
				}
			}
		}
		
		if (empty($request->get('id'))) {
			$chk_level_0 = Contents::where('type', $this->module)->whereNull('parent_id')->first();
			if (empty($chk_level_0)) {
				$re = Contents::create(['type' => $this->module]);
				if (empty($re))
					return self::response(['msg' => __('noti.error')], Response::HTTP_INTERNAL_SERVER_ERROR);

				$attr['parent_id'] = $re->id ?? null;
			} else
				$attr['parent_id'] = $chk_level_0->id ?? null;
		} else
			$attr['parent_id'] = Contents::where('type', $this->module)->whereNull('parent_id')->first()->id ?? null;
		
		$set = Contents::updateOrCreate(['id' => $request->get('id') ?? 0, 'type' => $this->module], $attr);

		if ($set->save()) {
			if (!empty($navbar['translate']['status']))
				self::setLangs($request, $set, $navbar['translate']['input']);

			if (!empty($navbar['contact']['status'])) {
				self::setContact($request, $set, $navbar['contact']['input']);
			}
			if (!empty($navbar['seo']['status']))
				SlugController::createMetadata($request, $set->id);

			if (!empty($navbar['other']['status']) || !empty($request->get('related'))) {
				foreach ($navbar['other']['input'] as $k_o => $o) {
					if (!empty($o['status'])) {
						if ($this->module == 'video' && $k_o == 'link') {
							$attr[$k_o] = $request->get($k_o);
						} else {
							$set->{$k_o}()->sync($request->get($k_o));
						}
					}
				}
				$set->related()->sync($request->get('related'));
			}

			if (!empty($request->get('pro'))) {
				$set->pro()->sync($request->get('pro'));
			}

			if (!empty($request->get('categories')))
				$set->categories()->sync($request->get('categories'));

			if (!empty($navbar['upload']['status']) && !empty($navbar['upload']['input']['gallery']['status'])) {
				$request->merge(['id' => $set->id]);
				self::save_gallery($request);
			}

			return self::response(['msg' => __('noti.success'), 'id' => $set->id, 'url' => route("admin.{$this->module}.{$this->method}.index")]);
		}

		return self::response(['msg' => __('noti.error')], Response::HTTP_INTERNAL_SERVER_ERROR);
	}

	/**
	 * Function : set translate
	 * Dev : Petch
	 * Update Date : 22 Jan 2025
	 */
	private function save_upload($request, $name, $type)
	{
		if (check_file_upload($request, $name, $type))
			return ['status' => false, 'msg' => __('noti.mime_image')];

		return ['status' => true, 'result' => self::set_image_upload($request, $name, "public/{$this->module}", "{$type}-")];
	}

	/**
	 * Function : set translate
	 * Dev : Petch
	 * Update Date : 22 Jan 2025
	 */
	private function setLangs(Request $request, $row, $config)
	{
		foreach (array_keys(languages()) as $lang) {
			$attr = [];
			foreach ($config as $key => $item)
				$attr[$key] = !empty($request->get("{$key}_{$lang}")) ? mwz_setTextString($request->get("{$key}_{$lang}"), $item['type'] == 'texteditor') : null;

			$row->langs()->updateOrCreate(['lang' => $lang], $attr);
		}
	}

	/**
	 * Function : save gallery
	 * Dev : Petch
	 * Update Date : 23 Jan 2025
	 */
	public function save_gallery(Request $request)
	{
		if (!empty($request->get('id'))) {
			$field_name = 'gallery';
			$upload_path = "/storage/{$this->module}/gallery";
			$images = $remove = [];
			// remove file if has
			if (!empty($request->get('id'))) {
				// delete file from multiple
				if ($request->get("{$field_name}_file_removed") != '') {
					$remove_list = json_decode($request->get("{$field_name}_file_removed"));
					if (!empty($remove_list)) {
						foreach ($remove_list as $rm_file) {
							$remove[] = $rm_file;
							$remove_file = str_replace('/storage', '', $rm_file);
							if (Storage::disk('public')->exists($remove_file))
								Storage::disk('public')->delete($remove_file);
						}
					}
				}
			}
			// upload file multiple
			$uploaded = [];
			if ($request->hasFile($field_name)) {
				foreach ($request->file($field_name) as $file) {
					if ($file->getClientMimeType() !== $file->getMimeType() || !in_array($file->getClientMimeType(), mime_type(['image'])))
						continue;

					$new_filename = $file->getClientOriginalName();
					$file->move(public_path($upload_path), $new_filename);
					$uploaded[] = $upload_path . '/' . $new_filename;
				}
			}

			// set image json 
			$index = 0;
			$file_lists = json_decode($request->get("{$field_name}_file_list"), 1);
			if (!empty($file_lists)) {
				foreach ($file_lists as $fl) {
					if ($fl['complete'] == false) {
						$image = $uploaded[$index++] ?? '';
						if (empty($image))
							continue;

						$fl['image'] =  $image;
						$fl['complete'] = true;
						$images[] = $fl;
					} elseif (!in_array($fl['image'], $remove))
						$images[] = $fl;
				}
			}

			$set = Contents::updateOrCreate(['id' => $request->get('id')], ['gallery' => !empty($images) ? json_encode($images, JSON_UNESCAPED_UNICODE) : null]);
			if ($set->save())
				return self::response(['msg' => __('noti.success'), 'id' => $set->id,  'url' => route("admin.{$this->module}.{$this->method}.index")]);

			return self::response(['msg' => __('noti.error'), 'id' => $request->get('id')], Response::HTTP_INTERNAL_SERVER_ERROR);
		}

		return self::response(['msg' => __('noti.error')], Response::HTTP_INTERNAL_SERVER_ERROR);
	}

	/**
	 * Function : set default
	 * Dev : Petch
	 * Update Date : 23 Jan 2025
	 */
	public function set_default(Request $request)
	{
		$limit = $this->config['setting']['limit_default'];
		$set = Contents::find($request->get('id'));
		$set->default = !$set->default;

		if (!empty($set->default) && !empty($limit['status']) && (Contents::where([['type', $this->module], ['default', 1]])->count() >= $limit['max']))
			return self::response(['msg' => __('noti.default.max', ['max' => $limit['max']])], Response::HTTP_BAD_REQUEST);
		elseif ($set->save())
			return self::response(['msg' => __('noti.success')]);

		return self::response(['msg' => __('noti.error')], Response::HTTP_INTERNAL_SERVER_ERROR);
	}

	/**
	 * Function : set status
	 * Dev : Petch
	 * Update Date : 23 Jan 2025
	 */
	public function set_status(Request $request)
	{
		$setting = $this->config['form']['sidebar']['setting'];
		$set = Contents::find($request->get('id'));
		$set->status = !$set->status;

		if (!empty($setting['status']) && !empty($setting['input']['categories']['status']) && $set->status && !$set->categories()->exists())
			return self::response(['msg' => __('noti.category')], Response::HTTP_BAD_REQUEST);

		if (!empty($this->config['form']['navbar']['seo']['status']))
			SlugController::slug_status($this->module, "{$this->module}_{$this->method}", $set->id, $set->status);

		if ($set->save())
			return self::response(['msg' => __('noti.success')]);

		return self::response(['msg' => __('noti.error')], Response::HTTP_INTERNAL_SERVER_ERROR);
	}

	/**
	 * Function : delete
	 * Dev : Petch
	 * Update Date : 23 Jan 2025
	 */
	public function set_delete(Request $request)
	{
		$set = Contents::find($request->get('id'));
		$gallery = json_decode($set->gallery, 1);
		$path_image = $set->image;
		$path_mobile = $set->mobile;
		$path_video = $set->video;

		if ($set->delete()) {
			self::delete_image($path_image);
			self::delete_image($path_mobile);
			self::delete_image($path_video);
			if (!empty($gallery)) {
				foreach ($gallery as $g)
					self::delete_image($g['image']);
			}

			if (!empty($this->config['form']['navbar']['seo']['status']))
				SlugController::remove_metadata($this->module,  "{$this->module}_{$this->method}", $request->get('id'));

			$limit = 0;
			if (!empty($this->config['setting']['limit']['status']))
				$limit = Contents::where('type', $this->module)->whereNotNull('parent_id')->count() < $this->config['setting']['limit']['max'];

			return self::response(['msg' => __('noti.delete'), 'limit' => $limit]);
		}

		return self::response(['msg' => __('noti.error')], Response::HTTP_BAD_REQUEST);
	}

	/**
	 * Function : set sort
	 * Dev : Petch
	 * Update Date : 23 Jan 2025
	 */
	public function set_sort(Request $request)
	{
		$node = Contents::find($request->get('id'));
		$move = $request->get('move') == 'up' ? $node->up() : $node->down();
		if ($move)
			return self::response(['msg' =>  __('noti.sort')]);

		return self::response(['msg' => __('noti.sort_error')], Response::HTTP_BAD_REQUEST);
	}

	/**
	 * Function : re order
	 * Dev : Poom
	 * Update Date : 29 Apr 2024
	 */
	public function re_order()
	{
		Contents::fixTree();
	}

	/**
	 * Function : get list
	 * Dev : Petch
	 * Update Date : 23 Jan 2025
	 */
	public function get_list(Request $request)
	{
		$filter = $request->get('parent');
		$req = Contents::where('type', $this->module)
			->whereNotNull('parent_id');

		if (!empty($filter)) {
			if (!empty($filter['search']))
				$req = $req->whereRelation('local', 'name', 'like', "%{$filter['search']}%");

			if (!empty($filter['id']))
				$req = $req->where('id', '<>', $filter['id']);

			if (!empty($filter['category_id']))
				$req = $req->whereRelation('category', 'id',  $filter['category_id']);
		}

		$data = $req->withDepth()
			->defaultOrder()
			->get();

		$result = [];
		foreach ($data as $list)
			$result[] = ['id' => $list->id, 'text' => $list->local->name, 'image' => $list->image ?? ''];

		return self::response(['results' => $result]);
	}

	/**
	 * Function : get category list
	 * Dev : Petch
	 * Update Date : 23 Jan 2025
	 */
	public function get_category_list(Request $request)
	{
		$filter = $request->get('parent');
		$req = ContentCategories::where('type', $this->module)
			->whereNotNull('parent_id');

		if (!empty($filter)) {
			if (!empty($filter['search']))
				$req = $req->whereRelation('local', 'name', 'like', "%{$filter['search']}%");
		}

		$data = $req->withDepth()
			->defaultOrder()
			->get();

		$result = [];
		foreach ($data as $list)
			$result[] = ['id' => $list->id, 'text' => str_repeat(' - ', $list->depth - 1) . $list->local->name, 'image' => $list->image ?? ''];

		return self::response(['results' => $result], 200);
	}


	/**
	 * Function : set translate
	 * Dev : Wan
	 * Update Date : 22 Jan 2025
	 */
	private function setOther(Request $request, $row, $config)
	{
		foreach ($config['other']['input'] as $k_o => $o) {
			if ($o['type'] == 'text') {
				$attr["{$k_o}"] = $request->get("{$k_o}");
				$row->updateOrCreate(['id' => $row->id], $attr);
			}else{
				if (!empty($o['status']))
				$row->{$k_o}()->sync($request->get($k_o));
			}
			
		}
	}

	/**
	 * Function : set translate
	 * Dev : Wan
	 * Update Date : 22 Jan 2025
	 */
	private function setContact(Request $request, $row, $config)
	{
		// Delete existing contacts first
		$row->contacts()->delete();

		// Get number of contact entries from first array
		$contactCount = count($request->get('name_contact', []));

		// Loop through each contact entry
		for ($i = 0; $i < $contactCount; $i++) {
			$contactData = [
				'name_contact' => $request->get('name_contact')[$i] ?? null,
				'detail_contact' => $request->get('detail_contact')[$i] ?? null, 
				'tel' => $request->get('tel')[$i] ?? null,
				'email' => $request->get('email')[$i] ?? null
			];

			// Only create contact if at least one field has data
			if (array_filter($contactData)) {
				$row->contacts()->create($contactData);
			}
		}
	}
}
