<?php

namespace Modules\Frontend\Livewire;

use Livewire\Component;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use App\Models\User;
use Illuminate\Auth\Notifications\ResetPassword;
use Illuminate\Support\Facades\Mail;
use Modules\Frontend\app\Emails\resetPassword as EmailsResetPassword;
use Modules\Member\app\Models\Member;

class LoginForm extends Component
{
    public $empCode = '';
    public $password = '';
    public $remember = false;
    public $showForgotPassword = false;
    public $resetEmail = '';

    protected $rules = [
        'empCode' => 'required|string',
        // อนุญาตเฉพาะตัวอักษรภาษาอังกฤษ ตัวเลข และอักขระพิเศษ (ASCII 33-126) ไม่รวมเว้นวรรค
        'password' => 'required|string|min:4|regex:/^[\x21-\x7E]+$/',
    ];

    protected $messages = [
        'empCode.required' => 'กรุณากรอกรหัสพนักงาน',
        'password.required' => 'กรุณากรอกรหัสผ่าน',
        'password.min' => 'รหัสผ่านต้องมีอย่างน้อย 4 ตัวอักษร',
        'password.regex' => 'รหัสผ่านต้องเป็นภาษาอังกฤษ ตัวเลข และอักขระพิเศษเท่านั้น',
    ];

    /**
     * Login function
     */
    public function login()
    {
        $this->validate();

        // ตรวจสอบผู้ใช้จากฐานข้อมูล
        $user = Member::where('code', $this->empCode)->where('status', 1)->first();

        if (!$user) {
            $this->dispatch('show-error', message: '❌ ไม่พบผู้ใช้งาน');
            return;
        }

        // ตรวจสอบสถานะระดับพนักงาน (lavel)
        if (empty($user->lavel) || empty($user->lavel->status) || $user->lavel->status == 0) {
            $this->dispatch('show-error', message: '❌ กลุ่มผู้ใช้งานนี้ยังไม่ได้เปิดใช้งาน');
            return;
        }

        // ตรวจสอบสถานะการใช้งาน
        if (!$user->status) {
            $this->dispatch('show-error', message: '⚠️ ผู้ใช้งานนี้ยังไม่ได้เปิดใช้งาน (Inactive)');
            return;
        }

        // ตรวจสอบรหัสผ่าน
        $password = Hash::make($this->password);
        // dd($password, $user->password);
        if (empty($user->password) || !Hash::check($this->password, $user->password)) {
            $this->dispatch('show-error', message: '❌ รหัสผ่านไม่ถูกต้อง');
            return;
        }

        // เข้าสู่ระบบสำเร็จ
        Auth::guard('member')->login($user, $this->remember);

        $this->dispatch('show-success', message: '');
        // Redirect ตาม role
        return $this->redirectBasedOnRole($user);
    }

    /**
     * Redirect based on user role
     */
    public static function redirectBasedOnRole($user)
    {
        if ($user->lavel_id === 2) {
            return redirect()->route('frontend.recorddatamanager');
        } elseif ($user->lavel_id === 3) {
            return redirect()->route('frontend.check');
        } elseif($user->lavel_id === 1) {
            return redirect()->route('frontend.recorddatabu');
        }
    }

    /**
     * Show forgot password popup
     */
    public function showForgotPasswordPopup()
    {
        $this->showForgotPassword = true;
        $this->resetEmail = '';
    }

    /**
     * Close forgot password popup
     */
    public function closeForgotPasswordPopup()
    {
        $this->showForgotPassword = false;
        $this->resetEmail = '';
    }

    /**
     * Send reset password link
     */
    public function sendResetLink()
    {
        $this->validate([
            'resetEmail' => 'required|email'
        ], [
            'resetEmail.required' => 'กรุณากรอกอีเมล',
            'resetEmail.email' => 'รูปแบบอีเมลไม่ถูกต้อง'
        ]);

        // ตรวจสอบว่ามีอีเมลนี้ในระบบหรือไม่
        $user = Member::where('email', $this->resetEmail)->first();
        if (!$user) {
            $this->dispatch('show-error', message: '❌ ไม่พบอีเมลนี้ในระบบ');
            return;
        }

        if ($user->status == 0) {
            $this->dispatch('show-error', message: '❌ อีเมลนี้ยังไม่ได้รับการยืนยัน');
            return;
        }

        $link = url('reset-password/' . $user->id);

        // ส่งอีเมล reset password (ยังไม่ได้ implement จริง)
        Mail::to($user->email)->send(new EmailsResetPassword('รีเซ็ตรหัสผ่าน', $link));
        // TODO: ส่งอีเมล reset password
        
        $this->dispatch('show-success', message: '✅ ส่งลิงก์ Reset Password ไปที่อีเมลของคุณแล้ว');
        $this->resetEmail = '';
        
        // ปิด popup หลังจาก 2 วินาที
        $this->dispatch('close-popup-after-delay');
    }


    public function render()
    {
        return view('frontend::livewire.login-form');
    }
}

