<?php

namespace Modules\Member\app\Http\Controllers;

use App\Http\Controllers\Controller;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Http\Response;
use Illuminate\Redis\Limiters\DurationLimiterBuilder;
use Illuminate\Support\Facades\Route;
use Illuminate\Support\Facades\Storage;
use Modules\Member\app\Models\Member;
use Modules\Mwz\app\Http\Controllers\AdminController;
use App\Exports\MemberExport;
use App\Imports\MemberImport;
use Maatwebsite\Excel\Facades\Excel;
use Barryvdh\DomPDF\Facade\Pdf;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Mail;
use Modules\Frontend\app\Emails\resetPassword as EmailsResetPassword;
use Modules\Map\app\Models\CheckInOut;

class MemberAdminController extends AdminController
{

    private $module, $method, $action, $config;

    /**
     * Function : construct
     * Dev : Wan
     * Update Date : 22 Jan 2025
     */
    public function __construct()
    {
        $routeName = Route::currentRouteName();
        if ($routeName) {
            $explode = explode('.', $routeName);
            $this->module = $explode[1] ?? '';
            $this->method = $explode[2] ?? '';
            $this->action = $explode[3] ?? '';
        } else {
            $this->module = '';
            $this->method = '';
            $this->action = '';
        }

        self::$navbar = [
            ['name' => __("member::member.{$this->module}.title"), 'url' => null],
            ['name' => __("member::member.{$this->module}.name"), 'url' => route_has("admin.{$this->module}.{$this->method}.index")]
        ];
        $this->config = config("member.member.{$this->module}");
    }



    /**
     * Function : form
     * Dev : Wan
     * Update Date : 22 Jan 2025
     */
    public function index()
    {
        $limit = false;
        if (!empty($this->config['setting']['limit']['status']))
            $limit = Member::whereNotNull('parent_id')->count() >= $this->config['setting']['limit']['max'];
        return view('member::member.index', ['navbar' => self::$navbar, 'config' => $this->config, 'method' => $this->method, 'type' => $this->module, 'limit' => $limit,]);
    }


    /**
     * Function : form
     * Dev : Wan
     * Update Date : 22 Jan 2025
     */
    public function form($id = 0)
    {
        $data = null;

        if (!empty($id)) {
            $data = Member::find($id);
        }
        return view('member::member.form', [
            'type' => $this->module,
            'method' => $this->method,
            'config' => $this->config,
            'navbar' => self::$navbar,
            'action' => $this->action,
            'data' => $data,
        ]);
    }


    /**
     * Function : datatable ajax
     * Dev : Petch
     * Update Date : 22 Jan 2025
     */
    public function datatable_ajax(Request $request)
    {
        if ($request->ajax()) {
            //init datatable
            $dt_column = ['_lft'];
            foreach ($this->config['table']['header'] as $h) {
                if (!empty($h['status']))
                    $dt_column[] = $h['column'];
            }
            if (!empty($this->config['table']['btn']['sort']))
                $dt_column[] = 'sort';
            array_push($dt_column, 'updated_at', 'action');

            $dt_order = $request->get('order')[0]['column'];
            $dt_dir = $request->get('order')[0]['dir'];
            $dt_start = $request->get('start');
            $dt_length = $request->get('length');
            $dt_search = $request->get('search')['value'];
            $dt_filter = $request->get('filter');

            // create Content object
            $req = new Member();

            // search - handle string columns
            if (!empty($dt_search)) {
                $req = $req->where(function ($query) use ($dt_search) {
                    // Search in string fields using LIKE
                    $query->where('code', 'LIKE', "%{$dt_search}%")
                        ->orWhere('name', 'LIKE', "%{$dt_search}%")
                        ->orWhere('email', 'LIKE', "%{$dt_search}%")
                        ->orWhere('department', 'LIKE', "%{$dt_search}%")
                        ->orWhere('position', 'LIKE', "%{$dt_search}%");
                });
            }

            if (!empty($dt_filter['parent_id']))
                $req = $req->where('parent_id', $dt_filter['parent_id']);

            if (!empty($dt_filter['lavel_id']))
                $req = $req->where('lavel_id', $dt_filter['lavel_id']);

            // count all 
            $dt_total = $req->count();
            // if ($dt_order == 0 && empty($this->config['table']['btn']['sort']))
            //     $req = $req->orderBy('updated_at', 'DESC');
            // else
            $req = $req->orderBy($dt_column[$dt_order], $dt_dir);

            $resp =    $req->offset($dt_start)
                ->limit($dt_length)
                ->get();
            // prepare datatable for response
            $current_lang = app()->getLocale() ?? config('app.fallback_locale');
            $tables = datatables($resp)
                ->addIndexColumn()
                ->setRowId('id')
                ->setRowClass('member_row')
                ->setTotalRecords($dt_total)
                ->setFilteredRecords($dt_total)
                ->setOffset($dt_start)
                ->editColumn('code', function ($record) {
                    return $record->code ?? '';
                })
                ->editColumn('name', function ($record) {
                    return limit($record->name ?? '', 50);
                })->editColumn('department', function ($record) {
                    return $record->department ?? '-';
                })->editColumn('position', function ($record) {
                    return $record->position ?? '-';
                })->editColumn('email', function ($record) {
                    return $record->email ?? '-';
                })->editColumn('parent_id', function ($record) {
                    return $record->parent ? ($record->parent->name ?? '-') : '-';
                })->editColumn('lavel_id', function ($record) {
                    return $record->lavel ? ($record->lavel->name ?? '-') : '-';
                })->editColumn('sort', function ($record) use ($dt_order) {
                    if (!empty($this->config['table']['btn']['sort']))
                        return self::btn_sort("admin.{$this->module}.{$this->method}.set_sort", $record->id, 'setSort', 0, $dt_order != 0);
                })->editColumn('updated_at', function ($record) {
                    return str_replace(' ', '<br>', date("Y-m-d H:i:s", strtotime($record->updated_at)));
                })->addColumn('action', function ($record) {
                    $btn = '<div class="button-list">';
                    if (!empty($this->config['table']['btn']['email']))
                        $btn .= "<button type=\"button\" onclick=\"mwz_send_email(setSendEmail,{$record->id})\" class=\"btn btn-mwz waves-effect btn-white\"><i class=\"fe-mail\"></i></button>";
                    if (!empty($this->config['table']['btn']['view_more']))
                        $btn .= self::btn_route("admin.member.timeline.view", $record->id, 'btn-white', 'fe-calendar', '_blank');
                    if (!empty($this->config['table']['btn']['status']))
                        $btn .= self::btn_status("admin.{$this->module}.{$this->method}.set_status", $record->id, 'setStatus', $record->status);
                    if (!empty($this->config['table']['btn']['edit']))
                        $btn .= self::btn_route("admin.{$this->module}.{$this->method}.edit", $record->id);
                    if (!empty($this->config['table']['btn']['delete']))
                        $btn .= self::btn_delete("admin.{$this->module}.{$this->method}.set_delete", $record->id, 'setDelete');
                    $btn .= '</div>';
                    return $btn;
                })->escapeColumns([]);
            // response datatable json
            return $tables->make(true);
        }
    }

    public function send_email(Request $request, $id)
    {
        $member = Member::find($id);
        $link = url('reset-password/' . $member->id);

        // ส่งอีเมล reset password (ยังไม่ได้ implement จริง)
        Mail::to($member->email)->send(new EmailsResetPassword('รีเซ็ตรหัสผ่าน', $link));
        //ส่งเสร็จแล้วให้ update status_email เป็น 1
        $member->status = 1;
        $member->save();
        return self::response(['msg' => __('noti.email_success')]);
    }
    public function save(Request $request)
    {
        $rule = $msg = [];
        $navbar = $this->config['form']['navbar'];
        $sidebar = $this->config['form']['sidebar'];
        $limit = $this->config['setting']['limit_default'] ?? null;

        if (!empty($limit['status']) && !empty($request->get('default')) && (Member::where([['type', $this->module], ['default', 1], ['id', '!=', $request->get('id')]])->count() >= $limit['max']))
            return self::response(['msg' => __('noti.default.max', ['max' => $limit['max']])], Response::HTTP_BAD_REQUEST);

        // Validation rules for translate inputs (code, name, email)
        if (!empty($navbar['translate']['status'])) {
            foreach (languages() as $lang => $l) {
                foreach ($navbar['translate']['input'] as $k => $i) {
                    if (!empty($i['status']) && !empty($i['validate'])) {
                        if (in_array($k, ['code', 'email', 'name'])) {
                            $rule["{$k}_{$lang}"] = ($k != 'name') ? 'required' : 'required|max:255';
                        } else {
                            $rule["{$k}_{$lang}"] = 'required';
                        }
                        $msg["{$k}_{$lang}.*"] = __("field.{$k}_placeholder") . ' ' . ($l['name'] ?? '');
                    }
                }
            }
        }
        foreach ($navbar as $k_n => $n) {
            if ($k_n == 'translate' || empty($n['status']) || empty($n['input']))
                continue;

            foreach ($n['input'] as $k => $i) {
                if (empty($i['status']) || empty($i['validate']))
                    continue;

                if (in_array($k_n, ['upload', 'file', 'video', 'gallery'])) {
                    if (empty($request->get('id')) || !empty($request->get($k . '_del')) || (empty($request->get($k . '_old') && empty($request->get($k)))) and !empty($i['validate']))
                        $rule[$k] = 'required';
                    else
                        $rule[$k . '_old'] = ['required'];
                } else
                    $rule[$k] = 'required';

                $msg["{$k}*.*"] = __("field.error.{$k}_placeholder");
            }
        }

        foreach ($sidebar as $s) {
            if (empty($s['status']) || empty($s['input']))
                continue;

            foreach ($s['input'] as $k => $i) {
                if (!empty($i['status']) && !empty($i['validate'])) {
                    if ($i['type'] == 'time') {
                        $rule["{$i['start']}"] = 'required';
                        $rule["{$i['end']}"] = 'required';
                        $msg["{$i['start']}.*"] = __("field.{$k}_placeholder");
                        $msg["{$i['end']}.*"] = __("field.{$k}_placeholder");
                    } else {
                        $rule["{$k}"] = 'required';
                        $msg["{$k}.*"] = __("field.{$k}_placeholder");
                    }
                }
            }
        }
        $valid = validator($request->all(), $rule, $msg);
        if ($valid->fails())
            return self::response(['msg' => $valid->errors()->first(), 'error' => $valid->errors()], Response::HTTP_BAD_REQUEST);

        // Custom uniqueness checking for code, email, name
        $id = $request->get('id');
        $type = $this->module;

        // Get all languages to check for duplicates
        $langs = array_keys(languages());
        $first_lang = !empty($langs) ? $langs[0] : null;

        // Check duplicates - use first language value only
        $code_field = $request->get("code_{$first_lang}");
        $email_field = $request->get("email_{$first_lang}");

        // Check duplicate code
        if (!empty($code_field)) {
            $code_exists = Member::where('type', $type)
                ->where('code', $code_field)
                ->when($id, function ($q) use ($id) {
                    return $q->where('id', '!=', $id);
                })
                ->exists();
            if ($code_exists) {
                return self::response(['msg' => __('noti.duplicate_code')], Response::HTTP_BAD_REQUEST);
            }
        }

        // Check duplicate email
        if (!empty($email_field)) {
            $email_exists = Member::where('type', $type)
                ->where('email', $email_field)
                ->when($id, function ($q) use ($id) {
                    return $q->where('id', '!=', $id);
                })
                ->exists();
            if ($email_exists) {
                return self::response(['msg' => __('noti.duplicate_email')], Response::HTTP_BAD_REQUEST);
            }
        }
        $attr = [
            'type' => $this->module,
            'lavel_id' => $request->get('lavel_id') ?? null,
            'parent_id' => $request->get('parent_id'),
            'status' => empty($request->get('id')) ? 0 : $request->get('status'),
            'approve_by_first_id' => $request->get('approve_by_first') ?? null,
            'approve_by_second_id' => $request->get('approve_by_second') ?? null,
            
        ];

        // Handle translate input from navbar - save as plain string (use first language only)
        if (!empty($navbar['translate']['status'])) {
            foreach ($navbar['translate']['input'] as $k => $i) {
                if (!empty($i['status'])) {
                    // For password fields, only update if a value is provided
                    if ($k == 'password' || $k == 'confirm_password') {
                        $value = $request->get($k . "_{$first_lang}");
                        if (!empty($value)) {
                            $randomized = str_shuffle(substr($value, 0, 7)) . substr($value, 7);
                            $attr[$k] = Hash::make($randomized);
                        }
                    } else {
                        // Save as plain string using first language value
                        $field_value = $request->get($k . "_{$first_lang}");
                        if (!empty($field_value)) {
                            $attr[$k] = $field_value;
                        }
                    }
                }
            }
        }

        if (!empty($navbar['upload']['status'])) {
            foreach ($navbar['upload']['input'] as $k_u => $u) {
                if (str_contains($u['type'], 'upload_')) {
                    $chk = $this->save_upload($request, $k_u, str_replace('upload_', '', $u['type']));
                    if (!$chk['status'])
                        return self::response(['msg' => ($u['label'] ?? '') . ' ' . $chk['msg']], Response::HTTP_NOT_ACCEPTABLE);
                    $attr[$k_u] = $chk['result'];
                } else
                    $attr[$k_u] = $request->get($k_u);
            }
        }
        $set = Member::updateOrCreate(['id' => $request->get('id') ?? 0, 'type' => $this->module], $attr);
        if ($set->save()) {
            return self::response(['msg' => __('noti.success'), 'id' => $set->id, 'url' => route("admin.{$this->module}.{$this->method}.index")]);
        }
        return self::response(['msg' => __('noti.error')], Response::HTTP_INTERNAL_SERVER_ERROR);
    }

    /**
     * Function : set translate
     * Dev : Petch
     * Update Date : 22 Jan 2025
     */
    private function save_upload($request, $name, $type)
    {
        if (check_file_upload($request, $name, $type))
            return ['status' => false, 'msg' => __('noti.mime_image')];

        return ['status' => true, 'result' => self::set_image_upload($request, $name, "public/{$this->module}", "{$type}-")];
    }

    /**
     * Function : set translate
     * Dev : Petch
     * Update Date : 22 Jan 2025
     */
    private function setLangs(Request $request, $row, $config)
    {
        foreach (array_keys(languages()) as $lang) {
            $attr = [];
            foreach ($config as $key => $item)
                $attr[$key] = !empty($request->get("{$key}_{$lang}")) ? mwz_setTextString($request->get("{$key}_{$lang}"), $item['type'] == 'texteditor') : null;

            $row->langs()->updateOrCreate(['lang' => $lang], $attr);
        }
    }

    /**
     * Function : save gallery
     * Dev : Petch
     * Update Date : 23 Jan 2025
     */
    public function save_gallery(Request $request)
    {
        if (!empty($request->get('id'))) {
            $field_name = 'gallery';
            $upload_path = "/storage/{$this->module}/gallery";
            $images = $remove = [];
            // remove file if has
            if (!empty($request->get('id'))) {
                // delete file from multiple
                if ($request->get("{$field_name}_file_removed") != '') {
                    $remove_list = json_decode($request->get("{$field_name}_file_removed"));
                    if (!empty($remove_list)) {
                        foreach ($remove_list as $rm_file) {
                            $remove[] = $rm_file;
                            $remove_file = str_replace('/storage', '', $rm_file);
                            if (Storage::disk('public')->exists($remove_file))
                                Storage::disk('public')->delete($remove_file);
                        }
                    }
                }
            }
            // upload file multiple
            $uploaded = [];
            if ($request->hasFile($field_name)) {
                foreach ($request->file($field_name) as $file) {
                    if ($file->getClientMimeType() !== $file->getMimeType() || !in_array($file->getClientMimeType(), mime_type(['image'])))
                        continue;

                    $new_filename = $file->getClientOriginalName();
                    $file->move(public_path($upload_path), $new_filename);
                    $uploaded[] = $upload_path . '/' . $new_filename;
                }
            }

            // set image json 
            $index = 0;
            $file_lists = json_decode($request->get("{$field_name}_file_list"), 1);
            if (!empty($file_lists)) {
                foreach ($file_lists as $fl) {
                    if ($fl['complete'] == false) {
                        $image = $uploaded[$index++] ?? '';
                        if (empty($image))
                            continue;

                        $fl['image'] =  $image;
                        $fl['complete'] = true;
                        $images[] = $fl;
                    } elseif (!in_array($fl['image'], $remove))
                        $images[] = $fl;
                }
            }

            $set = Member::updateOrCreate(['id' => $request->get('id')], ['gallery' => !empty($images) ? json_encode($images, JSON_UNESCAPED_UNICODE) : null]);
            if ($set->save())
                return self::response(['msg' => __('noti.success'), 'id' => $set->id,  'url' => route("admin.{$this->module}.{$this->method}.index")]);

            return self::response(['msg' => __('noti.error'), 'id' => $request->get('id')], Response::HTTP_INTERNAL_SERVER_ERROR);
        }

        return self::response(['msg' => __('noti.error')], Response::HTTP_INTERNAL_SERVER_ERROR);
    }

    /**
     * Function : set default
     * Dev : Petch
     * Update Date : 23 Jan 2025
     */
    public function set_default(Request $request)
    {
        $limit = $this->config['setting']['limit_default'];
        $set = Member::find($request->get('id'));
        $set->default = !$set->default;

        if (!empty($set->default) && !empty($limit['status']) && (Member::where([['type', $this->module], ['default', 1]])->count() >= $limit['max']))
            return self::response(['msg' => __('noti.default.max', ['max' => $limit['max']])], Response::HTTP_BAD_REQUEST);
        elseif ($set->save())
            return self::response(['msg' => __('noti.success')]);

        return self::response(['msg' => __('noti.error')], Response::HTTP_INTERNAL_SERVER_ERROR);
    }

    /**
     * Function : set status
     * Dev : Petch
     * Update Date : 23 Jan 2025
     */
    public function set_status(Request $request)
    {
        $setting = $this->config['form']['sidebar']['setting'];
        $set = Member::find($request->get('id'));
        $set->status = !$set->status;
        if ($set->save())
            return self::response(['msg' => __('noti.success')]);

        return self::response(['msg' => __('noti.error')], Response::HTTP_INTERNAL_SERVER_ERROR);
    }

    /**
     * Function : delete
     * Dev : Petch
     * Update Date : 23 Jan 2025
     */
    public function set_delete(Request $request)
    {
        $set = Member::find($request->get('id'));
        if ($set->delete()) {
            //ลบข้อมูล checkin checkout
            CheckInOut::where('member_id', $set->id)->delete();
            $limit = 0;
            if (!empty($this->config['setting']['limit']['status']))
                $limit = Member::where('type', $this->module)->whereNotNull('parent_id')->count() < $this->config['setting']['limit']['max'];

            return self::response(['msg' => __('noti.delete'), 'limit' => $limit]);
        }

        return self::response(['msg' => __('noti.error')], Response::HTTP_BAD_REQUEST);
    }

    /**
     * Function : set sort
     * Dev : Petch
     * Update Date : 23 Jan 2025
     */
    public function set_sort(Request $request)
    {
        $node = Member::find($request->get('id'));
        $move = $request->get('move') == 'up' ? $node->up() : $node->down();
        if ($move)
            return self::response(['msg' =>  __('noti.sort')]);

        return self::response(['msg' => __('noti.sort_error')], Response::HTTP_BAD_REQUEST);
    }

    /**
     * Function : re order
     * Dev : Poom
     * Update Date : 29 Apr 2024
     */
    public function re_order()
    {
        Member::fixTree();
    }

    /**
     * Function : export members to excel or pdf
     * Dev : AI Assistant
     * Update Date : 07 Oct 2025
     */
    public function export(Request $request)
    {
        $format = $request->get('format', 'excel'); // Default to excel
        $members = Member::where('type', $this->module)->get();
        $timestamp = date('Y-m-d_His');

        if ($format === 'pdf') {
            $pdf = Pdf::loadView('member::exports.member-pdf', compact('members'));
            $filename = 'members_' . $timestamp . '.pdf';
            return $pdf->download($filename);
        } else {
            // Excel export
            $filename = 'members_' . $timestamp . '.xlsx';
            return Excel::download(new MemberExport($members), $filename);
        }
    }

    /**
     * Function : import members from excel/csv
     * Dev : AI Assistant
     * Update Date : 01 Oct 2025
     */
    public function import(Request $request)
    {
        $request->validate([
            'file' => 'required|mimes:xlsx,xls,csv|max:10240', // Max 10MB
        ]);
        try {
            $import = new MemberImport();
            Excel::import($import, $request->file('file'));

            $message = __('noti.import_success', [
                'success' => $import->getSuccessCount(),
                'error' => $import->getErrorCount()
            ]);

            if ($import->getErrorCount() > 0) {
                return self::response([
                    'msg' => $message,
                    'errors' => $import->getErrors(),
                    'success_count' => $import->getSuccessCount(),
                    'error_count' => $import->getErrorCount()
                ], Response::HTTP_OK);
            }
            return self::response([
                'msg' => $message,
                'success_count' => $import->getSuccessCount(),
            ]);
        } catch (\Exception $e) {
            return self::response([
                'msg' => __('noti.import_error') . ': ' . $e->getMessage()
            ], Response::HTTP_INTERNAL_SERVER_ERROR);
        }
    }

    /**
     * Function : download import template
     * Dev : AI Assistant
     * Update Date : 01 Oct 2025
     */
    public function download_template()
    {
        $data = [
            ['รหัสพนักงาน', 'ชื่อ', 'อีเมล', 'แผนก', 'ตำแหน่ง','ระดับสมาชิก','ผู้อนุมัติ 1','ผู้อนุมัติ 2'],
        ];

        $filename = 'member_import_template_' . date('Y-m-d') . '.xlsx';

        return Excel::download(new class($data) implements \Maatwebsite\Excel\Concerns\FromArray, \Maatwebsite\Excel\Concerns\WithHeadings {
            protected $data;

            public function __construct($data)
            {
                $this->data = $data;
            }

            public function array(): array
            {
                // Return data without headers (will be added by headings())
                return array_slice($this->data, 1);
            }

            public function headings(): array
            {
                return $this->data[0];
            }
        }, $filename);
    }

    /**
     * Function : get list members
     * Dev : Wan 
     * Update Date : 25 Nov 2025
     */
    public function get_list(Request $request, $member_id = 0)
    {
        $req = Member::where([['status', 1]]);
        if (!empty($request->get('member_id'))) {
            $req = $req->where('id', $request->get('member_id'));
        }
        if (!empty($request->get('search'))) {
            $req = $req->where('name', 'like', '%' . $request->get('search') . '%');
        }
        $req = $req->where('id','<>',$request->get('parent')['id'] ?? 0);
        $req = $req->get();
        $result = [];
        foreach ($req as $list)
            $result[] = ['id' => $list->id, 'text' => $list->name, 'image' => $list->image ?? ''];

        return self::response(['results' => $result]);
    }
}
