<?php

namespace Modules\Member\app\Http\Controllers;

use App\Http\Controllers\Controller;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Http\Response;
use Illuminate\Support\Facades\Route;
use Modules\Member\app\Models\Member;
use Modules\Member\app\Models\MemberLavel;
use Modules\Mwz\app\Http\Controllers\AdminController;
use Modules\Mwz\app\Http\Controllers\SlugController;

class MemberlavelAdminController extends AdminController
{
    private $module, $method, $action, $config;

    /**
     * Function : construct
     * Dev : Wan
     * Update Date : 23 Jan 2025
     */
    public function __construct()
    {

        $routeName = Route::currentRouteName();
		if ($routeName) {
			$explode = explode('.', $routeName);
			$this->module = $explode[1] ?? '';
			$this->method = $explode[2] ?? '';
			$this->action = $explode[3] ?? '';
		} else {
			$this->module = '';
			$this->method = '';
			$this->action = '';
		}
        self::$navbar = [
            ['name' => __("member::lavel.{$this->module}.title"), 'url' => null],
            ['name' => __("member::lavel.{$this->module}.name"), 'url' => route_has("admin.{$this->module}.{$this->method}.index")]
        ];

        $this->config = config("member.lavel.{$this->module}");
    }

    public function index()
    {
        $limit = false;
        if (!empty($this->config['setting']['limit']['status']))
            $limit = MemberLavel::where('type', $this->module)->whereNotNull('parent_id')->count() >= $this->config['setting']['limit']['max'];
        return view('member::lavel.index', [
            'navbar' => self::$navbar,
            'method' => $this->method,
            'config' => $this->config,
            'limit' => $limit,
            'type' => $this->module,
        ]);
    }

    /**
     * Function : construct
     * Dev : Wan
     * Update Date : 23 Jan 2025
     */
    public function datatable_ajax(Request $request)
    {
        if ($request->ajax()) {
            //init datatable
            $dt_column = ['_lft'];
            foreach ($this->config['table']['header'] as $h) {
                if (!empty($h['status']))
                    $dt_column[] = $h['column'];
            }
            if (!empty($this->config['table']['btn']['sort']))
                $dt_column[] = 'sort';
            array_push($dt_column, 'updated_at', 'action');

            $dt_order = $request->get('order')[0]['column'];
            $dt_dir = $request->get('order')[0]['dir'];
            $dt_start = $request->get('start');
            $dt_length = $request->get('length');
            $dt_search = $request->get('search')['value'];
            $dt_filter = $request->get('filter');
            // create Category object
            $req = MemberLavel::where('type', $this->module);

            // dt_search 
            if (!empty($dt_search))
                $req = $req->where('name', 'like', "%{$dt_search}%");

            // dt_filter
            if (!empty($dt_filter['parent_id']))
                $req = $req->whereDescendantOrSelf($dt_filter['parent_id']);

            // count all 
            $dt_total = $req->count();

            // set query order & limit from datatable
            $resp = $req->orderBy($dt_column[$dt_order], $dt_dir)
                ->offset($dt_start)
                ->limit($dt_length)
                ->withDepth()
                ->defaultOrder()
                ->get();
            // prepare datatable for response
            $tables = datatables($resp)
                ->addIndexColumn()
                ->setRowId('id')
                ->setRowClass('content_row')
                ->setTotalRecords($dt_total)
                ->setFilteredRecords($dt_total)
                ->setOffset($dt_start)
                ->editColumn('image', function ($record) {
                    return $this->view_file('image', $record->image);
                })->editColumn('name', function ($record) {
                    return $record->name ?? '';
                })->editColumn('parent_id', function ($record) {
                    // return self::make_name($record->parent, $record->parent->local->name ?? '');
                })->editColumn('sort', function ($record) use ($dt_order) {
                    if (!empty($this->config['table']['btn']['sort']))
                        return self::btn_sort("admin.{$this->module}.{$this->method}.set_sort", $record->id, 'setSort', 0, $dt_order != 0);
                })->editColumn('updated_at', function ($record) {
                    return str_replace(' ', '<br>', date("Y-m-d H:i:s", strtotime($record->updated_at)));
                })->addColumn('action', function ($record) {
                    $btn = '<div class="button-list">';

                    if (!empty($this->config['table']['btn']['default']))
                        $btn .= self::btn_default("admin.{$this->module}.{$this->method}.set_default", $record->id, 'setDefault', $record->default);
                    if (!empty($this->config['table']['btn']['status']))
                        $btn .= self::btn_status("admin.{$this->module}.{$this->method}.set_status", $record->id, 'setStatus', $record->status);
                    if (!empty($this->config['table']['btn']['edit']))
                        $btn .= self::btn_route("admin.{$this->module}.{$this->method}.edit", $record->id);
                    if (!empty($this->config['table']['btn']['delete']))
                        $btn .= self::btn_delete("admin.{$this->module}.{$this->method}.set_delete", $record->id, 'setDelete');

                    $btn .= '</div>';

                    return $btn;
                })->escapeColumns([]);
            // response datatable json
            return $tables->make(true);
        }
    }

    /**
     * Function : construct
     * Dev : Wan
     * Update Date : 23 Jan 2025
     */
    public function form($id = 0)
    {
        self::$navbar[] =  ['name' =>  __("action.{$this->action}"), 'url' => null];
        $meta = self::set_metadata($this->module,  "{$this->module}_{$this->method}", 3);
        $data = null;
        $limit = false;
        if (!empty($id)) {
            $data = MemberLavel::where([['id', $id], ['type', $this->module]])->first();
            if (empty($data))
                return abort(Response::HTTP_NOT_FOUND);

            if (!empty($data->langs)) {
                $data->langs = $data->langs->groupBy('lang')->map(function ($row) {
                    return  $row->first();
                });
            }
            if (!empty($this->config['form']['navbar']['seo']['status']))
                $meta = array_merge($meta, SlugController::getMetadata($this->module, "{$this->module}_{$this->method}", $id));
        } elseif (!empty($this->config['setting']['limit']['status']))
            $limit = MemberLavel::where('type', $this->module)->whereNotNull('parent_id')->count() >= $this->config['setting']['limit']['max'];

        return view('member::lavel.form', [
            'data' => $data,
            'navbar' => self::$navbar,
            'type' => $this->module,
            'method' => $this->method,
            'meta' => $meta,
            'config' => $this->config,
            'limit' => $limit,
            'action' => $this->action
        ]);
    }

/**
     * Function : construct
     * Dev : Wan
     * Update Date : 23 Jan 2025
     */
    public function save(Request $request)
    {
        $rule = $msg = [];
        $navbar = $this->config['form']['navbar'];
        $sidebar = $this->config['form']['sidebar'];
        $limit = $this->config['setting']['limit_default'] ?? null;
        if (!empty($limit['status']) && !empty($request->get('default')) && (MemberLavel::where([['type', $this->module], ['default', 1], ['id', '!=', $request->get('id')]])->count() >= $limit['max']))
            return self::response(['msg' => __('noti.default.max', ['max' => $limit['max']])], Response::HTTP_BAD_REQUEST);

        if (!empty($navbar['translate']['status'])) {
            foreach (languages() as $lang => $l) {
                foreach ($navbar['translate']['input'] as $k => $i) {
                    if (!empty($i['status']) && !empty($i['validate'])) {
                        $rule["{$k}_{$lang}"] = ($k != 'name') ? 'required' : 'required|max:255';
                        $msg["{$k}_{$lang}.*"] = __("field.{$k}_placeholder") . ' ' . ($l['name'] ?? '');
                    }
                }
            }
        }
        foreach ($navbar as $k_n => $n) {
            if ($k_n == 'translate' || empty($n['status']) || empty($n['input']))
                continue;

            foreach ($n['input'] as $k => $i) {
                if (empty($i['status']) || empty($i['validate']))
                    continue;

                if (in_array($k_n, ['upload', 'file', 'video', 'gallery'])) {
                    if (empty($request->get('id')) || !empty($request->get($k . '_del')) || (empty($request->get($k . '_old') && empty($request->get($k)))) and !empty($i['validate']))
                        $rule[$k] = 'required';
                    else
                        $rule[$k . '_old'] = ['required'];
                } else
                    $rule[$k] = 'required';

                $msg["{$k}*.*"] = __("field.error.{$k}_placeholder");
            }
        }
        foreach ($sidebar as $s) {
            if (empty($s['status']) || empty($s['input']))
                continue;

            foreach ($s['input'] as $k => $i) {
                if (!empty($i['status']) && !empty($i['validate'])) {
                    if ($i['type'] == 'time') {
                        $rule["{$i['start']}"] = 'required';
                        $rule["{$i['end']}"] = 'required';
                        $msg["{$i['start']}.*"] = __("field.{$k}_placeholder");
                        $msg["{$i['end']}.*"] = __("field.{$k}_placeholder");
                    } else {
                        $rule["{$k}"] = 'required';
                        $msg["{$k}.*"] = __("field.{$k}_placeholder");
                    }
                }
            }
        }

        $valid = validator($request->all(), $rule, $msg);
        if ($valid->fails())
            return self::response(['msg' => $valid->errors()->first(), 'error' => $valid->errors()], Response::HTTP_BAD_REQUEST);

        $attr = [
            'type' => $this->module,
            'status' => 1
        ];

        // Handle translate input from navbar
        if (!empty($navbar['translate']['status'])) {
            foreach ($navbar['translate']['input'] as $k => $i) {
                if (!empty($i['status'])) {
                    // For direct fields (not translated)
                    $attr[$k] = $request->get($k . "_{$lang}");
                }
            }
        }

        if (!empty($navbar['upload']['status'])) {
            foreach ($navbar['upload']['input'] as $k_u => $u) {
                if (str_contains($u['type'], 'upload_')) {
                    $chk = $this->save_upload($request, $k_u, str_replace('upload_', '', $u['type']));
                    if (!$chk['status'])
                        return self::response(['msg' => ($u['label'] ?? '') . ' ' . $chk['msg']], Response::HTTP_NOT_ACCEPTABLE);
                    $attr[$k_u] = $chk['result'];
                } else
                    $attr[$k_u] = $request->get($k_u);
            }
        }

        $set = MemberLavel::updateOrCreate(['id' => $request->get('id') ?? 0, 'type' => $this->module], $attr);

        if ($set->save()) {
            return self::response(['msg' => __('noti.success'), 'id' => $set->id, 'url' => route("admin.{$this->module}.{$this->method}.index")]);
        }

        return self::response(['msg' => __('noti.error')], Response::HTTP_INTERNAL_SERVER_ERROR);
    }

    /**
     * Function : set translate
     * Dev : Wan
     * Update Date : 22 Jan 2025
     */
    private function save_upload($request, $name, $type)
    {
        if (check_file_upload($request, $name, $type))
            return ['status' => false, 'msg' => __('noti.mime_image')];

        return ['status' => true, 'result' => self::set_image_upload($request, $name, "public/{$this->module}", "{$type}-")];
    }

    /**
     * Function : set translate
     * Dev : Wan
     * Update Date : 23 Jan 2025
     */
    private function setLangs(Request $request, $row, $config)
    {
        foreach (array_keys(languages()) as $lang) {
            $attr = [];
            foreach ($config as $key => $item)
                $attr[$key] = !empty($request->get("{$key}_{$lang}")) ? mwz_setTextString($request->get("{$key}_{$lang}"), $item['type'] == 'texteditor') : null;

            $row->langs()->updateOrCreate(['lang' => $lang], $attr);
        }
    }

    /**
     * Function : set default
     * Dev : Wan
     * Update Date : 23 Jan 2025
     */
    public function set_default(Request $request)
    {
        $limit = $this->config['setting']['limit_default'];
        $set = MemberLavel::find($request->get('id'));
        $set->default = !$set->default;

        if (!empty($set->default) && !empty($limit['status']) && (MemberLavel::where([['type', $this->module], ['default', 1]])->count() >= $limit['max']))
            return self::response(['msg' => __('noti.default.max', ['max' => $limit['max']])], Response::HTTP_BAD_REQUEST);
        elseif ($set->save())
            return self::response(['msg' => __('noti.success')]);

        return self::response(['msg' => __('noti.error')], Response::HTTP_INTERNAL_SERVER_ERROR);
    }

    /**
     * Function : set status
     * Dev : Wan
     * Update Date : 23 Jan 2025
     */
    public function set_status(Request $request)
    {
        $set = MemberLavel::find($request->get('id'));
        $set->status = !$set->status;

        if (!empty($this->config['form']['navbar']['seo']['status']))
            SlugController::slug_status($this->module, "{$this->module}_{$this->method}", $set->id, $set->status);

        if ($set->save())
            return self::response(['msg' => __('noti.success')]);

        return self::response(['msg' => __('noti.error')], Response::HTTP_INTERNAL_SERVER_ERROR);
    }

    /**
     * Function : delete
     * Dev : Wan
     * Update Date : 23 Jan 2025
     */
    public function set_delete(Request $request)
    {
        $set = MemberLavel::find($request->get('id'));
        $path_image = $set->image;

        // Replace: $set->contents()->exists() and update
        // Find and deactivate all related content by member_lavel_id (assumption: related table/model: MemberLavelContent)
        Member::where('lavel_id', $set->id)->update(['status' => 0]);

        if ($set->delete()) {
            self::delete_image($path_image);
            self::re_order();

            if (!empty($this->config['form']['navbar']['seo']['status']))
                SlugController::remove_metadata($this->module,  "{$this->module}_{$this->method}", $request->get('id'));

            $limit = 0;
            if (!empty($this->config['setting']['limit']['status']))
                $limit = MemberLavel::where('type', $this->module)->whereNotNull('parent_id')->count() < $this->config['setting']['limit']['max'];

            return self::response(['msg' => __('noti.delete'), 'limit' => $limit]);
        }

        return self::response(['msg' => __('noti.error')], Response::HTTP_BAD_REQUEST);
    }

    /**
     * Function : set sort
     * Dev : Wan
     * Update Date : 23 Jan 2025
     */
    public function set_sort(Request $request)
    {
        $node = MemberLavel::find($request->get('id'));
        $move = $request->get('move') == 'up' ? $node->up() : $node->down();

        if ($move)
            return self::response(['msg' =>  __('noti.sort')]);

        return self::response(['msg' => __('noti.sort_error')], Response::HTTP_BAD_REQUEST);
    }

    /**
     * Function : re order
     * Dev : Wan
     * Update Date : 23 Jan 2025
     */
    public static function re_order()
    {
        MemberLavel::fixTree();
    }

    /**
     * Function : get list
     * Dev : Wan
     * Update Date : 23 Jan 2025
     */
    public function get_list(Request $request)
    {
        $filter = $request->get('parent');
        $req = MemberLavel::where('type', $this->module);
        if (!empty($filter)) {
            if (!empty($filter['search']))
                $req = $req->whereRelation('local', 'name', 'like', "%{$filter['search']}%");
            // if (!empty($filter['id']))
            //     $req = $req->where('id', '<>', $filter['id']);
        }
        // if (!empty($this->config['setting']['limit_parent']['status']))
        //     $req = $req->having('depth', '<=', $this->config['setting']['limit_parent']['level']);

        $data = $req->withDepth()
            ->defaultOrder()
            ->get();
        $result = [];
        foreach ($data as $list)
            $result[] = ['id' => $list->id, 'text' => $list->name, 'image' => $list->image ?? ''];
        return self::response(['results' => $result], 200);
    }
}
