<?php

namespace Modules\Mwz\app\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\File;
use Modules\Mwz\app\Models\AdminMenus;

class AdminMenuController extends AdminController
{
	/**
	 * Function : construct
	 * Dev : Poom
	 * Update Date : 26 Apr 2024
	 */
	public function __construct()
	{
		$this->middleware('auth:admin');
		self::$navbar = [
			['name' => __('mwz::menu.title'), 'url' => null],
			['name' => __('mwz::menu.name'), 'url' => route_has('admin.mwz.admin_menu.index')]
		];
	}

	/**
	 * Function : index
	 * Dev : Poom
	 * Update Date : 26 Apr 2024
	 * @return view of menu index
	 */
	public function index()
	{
		return view('mwz::menu.index', ['navbar' => self::$navbar]);
	}

	/**
	 * Function : datatable_ajax
	 * Dev : Poom
	 * Update Date : 26 Apr 2024
	 * @param Request $request
	 * @return json of datatable
	 */
	public function datatable_ajax(Request $request,)
	{
		if ($request->ajax()) {
			//init datatable
			$dt_column = ['_lft', 'icon', 'name',  'link', 'sort', 'updated_at', 'action'];

			$dt_order = $request->get('order')[0]['column'];
			$dt_dir = $request->get('order')[0]['dir'];
			$dt_start = $request->get('start');
			$dt_length = $request->get('length');
			$dt_search = $request->get('search')['value'];
			$dt_filter = $request->get('filter');

			// create AdminMenus object
			$req = new AdminMenus();
			// parent_id
			if (!empty($dt_filter['parent_id'])) {
				$req = $req->where(function ($query) use ($dt_filter) {
					$query->where('parent_id', $dt_filter['parent_id'])
						->orwhere('id', $dt_filter['parent_id']);
				});
			}
			if (!empty($dt_filter['section'])) {
				$req = $req->where('section', $dt_filter['section']);
			}
			// dt_search 
			if (!empty($dt_search)) {
				$req = $req->where(function ($query) use ($dt_search) {
					$query->orWhere('name', 'like', "%$dt_search%");
				});
			}

			// count all AdminMenus
			$dt_total = $req->count();
			// set query order & limit from datatable
			$data = $req->orderBy($dt_column[$dt_order], $dt_dir)
				->offset($dt_start)->limit($dt_length)
				->withDepth()->defaultOrder()->get();

			// prepare datatable for response
			$tables = datatables($data)
				->addIndexColumn()
				->setRowId('id')
				->setRowClass('menu_row')
				->setTotalRecords($dt_total)
				->setFilteredRecords($dt_total)
				->setOffset($dt_start)
				->editColumn('updated_at', function ($record) {
					return str_replace(' ', '<br>', date("Y-m-d H:i:s", strtotime($record->updated_at)));
				})->editColumn('name', function ($record) {
					return str_implode(' &raquo; ', self::parent($record));
				})->addColumn('link', function ($record) {
					if (!empty($record->permission->route_name)) {
						$route = $record->permission->route_name;
						$link = route_has($route);
						return "<a class=\"text-decoration-underline\" href=\"$link\" target=\"_blank\">$route</a>";
					} else {
						return null;
					}
				})->editColumn('icon', function ($record) {
					return !empty($record->icon) ? ('<i class="' . ($record->icon) . ' fs-3" ></i>') : '';
				})->editColumn('sort', function ($record) use ($dt_order) {
					return self::btn_sort("admin.mwz.admin_menu.set_sort", $record->id, 'setSort', 0, $dt_order != 0);
				})->addColumn('action', function ($record) {
					$btn = '<div class="button-list">';
					$btn .= self::btn_status('admin.mwz.admin_menu.set_status', $record->id, 'setStatus', $record->status);
					$btn .= self::btn_route('admin.mwz.admin_menu.edit', $record->id);
					$btn .= self::btn_delete('admin.mwz.admin_menu.set_delete', $record->id, 'setDelete');
					$btn .= '</div>';
					return $btn;
				})
				->escapeColumns([]);
			// response datatable json
			return $tables->make(true);
		}
	}

	/**
	 * Function : page form
	 * Dev : Poom
	 * Update Date : 26 Apr 2024
	 * @param int $id
	 * @return view of menu form
	 */
	public function form($id = 0)
	{
		self::$navbar[] =  ['name' => !empty($id) ? __('action.edit') : __('action.add'), 'url' => null];
		$data = null;
		if (!empty($id)) {
			$data = AdminMenus::find($id);
		}

		return view('mwz::menu.form', ['data' => $data, 'navbar' => self::$navbar]);
	}

	/**
	 * Function : save
	 * Dev : Poom
	 * Update Date : 26 Apr 2024
	 * @param Request $request
	 * @return json of save status
	 */
	public function save(Request $request)
	{
		$valid = validator($request->all(), [
			'name' => 'required|max:255',
		], [
			'name.*' => __('mwz::menu.field.name_placeholder')
		]);

		if ($valid->fails())
			return self::response(['msg' => $valid->errors()->first(), 'error' => $valid->errors()], 404);

		$attr = [
			"name" => $request->get('name'),
			"icon" => $request->get('icon'),
			"permission_id" => $request->get('permission_id'),
			"parent_id" => $request->get('parent_id'),
			"status" => $request->get('status') ?? 0
		];

		if (empty($request->get('id')))
			$attr["sequence"] = AdminMenus::max('sequence') + 1;

		$set = AdminMenus::updateOrCreate(['id' => $request->get('id') ?? 0], $attr);
		if ($set->save()) {
			self::re_order();
			self::set_menu();
			return self::response(['msg' => __('noti.success')],);
		}

		return self::response(['msg' => __('noti.error')], 404);
	}

	/**
	 * Function : set_status
	 * Dev : Poom
	 * Update Date : 26 Apr 2024
	 * @param Request $request
	 * @return json of set status
	 */
	public function set_status(Request $request)
	{
		$set = AdminMenus::find($request->get('id'));
		$set->status = !$set->status;
		if ($set->save()) {
			self::set_menu();
			return self::response(['msg' => __('noti.success')], 200);
		}
		return self::response(['msg' => __('noti.error')], 404);
	}

	/**
	 * Function : set_delete
	 * Dev : Poom
	 * Update Date : 26 Apr 2024
	 * @param Request $request
	 * @return json of set delete
	 */
	public function set_delete(Request $request)
	{
		$set = AdminMenus::find($request->get('id'));
		if ($set->delete()) {
			self::re_order();
			self::set_menu();
			return self::response(['msg' => __('noti.delete')], 200);
		}
		return self::response(['msg' => __('noti.error')], 404);
	}

	/**
	 * Function : set_sort
	 * Dev : Peh
	 * Update Date : 26 Apr 2024
	 * @param Request $request
	 * @return json of set sort
	 */
	public function set_sort(Request $request)
	{
		$node = AdminMenus::find($request->get('id'));
		$move = false;
		if ($request->get('move') == 'up') {
			$move = $node->up();
		} else {
			$move = $node->down();
		}
		if ($move) {
			self::re_order();
			self::set_menu();
			return self::response(['msg' => __('noti.success')], 200);
		}

		return self::response(['msg' => __('noti.error')], 404);
	}

	/**
	 * Function : re order
	 * Dev : Peh
	 * Update Date : 26 Apr 2024
	 * @return void
	 */
	public function re_order()
	{
		$all = AdminMenus::orderBy('parent_id')->orderBy('_lft')->orderBy('updated_at')->get();
		if (!empty($all)) {
			foreach ($all  as $cnt => $cat) {
				$cat->sequence = $cnt + 1;
				$cat->save();
			}
		}
		AdminMenus::fixTree();
	}

	/**
	 * Function : tree
	 * Dev : Peh
	 * Update Date : 26 Apr 2024
	 * @return view of menu tree
	 */
	public function tree()
	{
		self::$navbar[] =  ['name' => __('action.sort'), 'url' => null];
		$data = AdminMenus::where('status', 1)->orderBy('_lft')->get()->toTree();

		return view('mwz::menu.tree', ['navbar' => self::$navbar, 'data' => $data]);
	}


	/**
	 * Function : tree sort product category
	 * Dev : Peh
	 * Update Date : 27 Oct 2023`+````````````````
	 * @param POST
	 * @return json of tree sort status
	 */
	public function tree_sort(Request $request)
	{
		try {
			//code...
			if (empty($request->id)) {
				return self::response(['msg' => 'no id'], 404);
			} else {
				$menu = AdminMenus::find($request->id);
				if (empty($menu)) {
					return self::response(['msg' => __('noti.no-data')], 404);
				} else {
					$parent = AdminMenus::find($request->parent_id);
					if (empty($parent)) {
						$menu->parent_id = null;
						$neighbor = AdminMenus::where('parent_id', null)->orderBy('_lft', 'asc')->get();
						if ($neighbor->count() > $request->position && isset($neighbor[$request->position])) {
							if ($request->position > $request->old_position) {
								$menu =  $menu->afterNode($neighbor[$request->position]);
							} else {
								$menu =  $menu->beforeNode($neighbor[$request->position]);
							}
						} else {
							// If position is out of bounds, just save as root
							$menu->parent_id = null;
						}
						$menu->save();
						$resp = ['msg' => __('noti.success'), 'data' => $neighbor];
					} else {
						$main_id = $parent->id;
						$old_parent_id = $menu->parent_id;
						$neighbor = AdminMenus::where('parent_id', $main_id)->orderBy('_lft', 'asc')->get();
						if (empty($neighbor)) {
							$menu->parent_id = $main_id;
							$menu->save();
							$resp = ['msg' => __('noti.success')];
						} else {
							if ($old_parent_id == $main_id) {
								if ($neighbor->count() > $request->position && isset($neighbor[$request->position])) {
									if ($request->position > $request->old_position) {
										$menu = $menu->afterNode($neighbor[$request->position]);
									} else {
										$menu = $menu->beforeNode($neighbor[$request->position]);
									}
								} else {
									// Position out of bounds, just save with parent
									$menu->parent_id = $main_id;
								}
							} else {
								if ($neighbor->count() > $request->position && isset($neighbor[$request->position])) {
									$menu = $menu->beforeNode($neighbor[$request->position]);
								} else {
									// Position out of bounds, just save with parent
									$menu->parent_id = $main_id;
								}
							}
							$menu->save();
							$resp = ['msg' => __('noti.success'), 'data' => $neighbor];
						}
					}
				}
			}
			self::set_menu();
			return self::response($resp);
		} catch (\Throwable $th) {
			return self::response(['msg' => __('noti.error')], 404);
		}
	}

	/**
	 * Function : getRootNode
	 * Dev : Peh
	 * Update Date : 26 Apr 2024
	 * @return object of root node
	 */
	public function getRootNode()
	{
		return AdminMenus::where('parent_id', null)->first();
	}

	/**
	 * Function : icons
	 * Dev : Poom
	 * Update Date : 26 Apr 2024
	 * @return json of icons
	 */
	public function icons()
	{
		return File::json("assets/json/icons.json", JSON_UNESCAPED_UNICODE);
	}

	/**
	 * Function : get_list
	 * Dev : Poom
	 * Update Date : 26 Apr 2024
	 * @param Request $request
	 * @return json of menu list
	 */
	public function get_list(Request $request)
	{
		$filter = $request->get('parent');
		$req = new AdminMenus();
		$req = $req->withDepth()->defaultOrder();
		if (!empty($filter)) {
			$req = $req->where(function ($q) use ($filter) {
				if (!empty($filter['search'])) {
					$q = $q->where('name', 'like', "%{$filter['search']}%");
				}
				if (!empty($filter['id'])) {
					$q = $q->where('id', '!=',  $filter['id']);
					$q = $q->having('depth', '<=', 1);
				}
				return $q;
			});
		}
		$req = $req->orderBy('sequence')->get();
		$result = [];
		foreach ($req as $list) {
			$result[] = ['id' => $list->id, 'text' => str_implode(' &raquo; ', self::parent($list)), 'image' => $list->image ?? ''];
		}

		return self::response(['results' => $result]);
	}

	private static function set_menu()
	{
		$row = AdminMenus::select(['id', 'icon', 'name', 'permission_id', '_lft'])
			->with(['child', 'permission'])
			->where(function ($query) {
				$query->where('status', 1)->whereNull('parent_id');
			})->orderBy('_lft')->get();

		File::put('assets/json/menu.json', json_encode($row, JSON_UNESCAPED_UNICODE));
	}
}
