<?php

namespace Modules\Mwz\app\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Http\Response;
use Illuminate\Support\Facades\File;
use Modules\Mwz\app\Models\Cities;
use Modules\Mwz\app\Models\CityTranslate;
use Modules\Mwz\app\Models\Districts;

class CityAdminController extends AdminController
{
    /**
     * Function : construct
     * Dev : Poom
     * Update Date : 26 Apr 2024
     */
    public function __construct()
    {
        $this->middleware('auth:admin');

        self::$navbar = [
            ['name' => __("mwz::city.title"), 'url' => null],
            ['name' => __("mwz::city.name"), 'url' => route_has("admin.mwz.city.index")]
        ];
    }

    /**
     * Function : index
     * Dev : Poom
     * Update Date : 26 Apr 2024
     */
    public function index()
    {
        return view('mwz::city.index', ['navbar' => self::$navbar]);
    }

    /**
     * Function : datatable_ajax
     * Dev : Poom
     * Update Date : 26 Apr 2024
     */
    public function datatable_ajax(Request $request)
    {
        if ($request->ajax()) {
            //init datatable
            $dt_column = ['id', 'province_id', 'district_id', 'name', 'zipcode',    'updated_at', 'action'];
            $dt_order = $request->get('order')[0]['column'];
            $dt_dir = $request->get('order')[0]['dir'];
            $dt_start = $request->get('start');
            $dt_length = $request->get('length');
            $dt_search = $request->get('search')['value'];
            $dt_filter = $request->get('filter');

            // create Cities object
            $req = new Cities();
            if (!empty($dt_filter['province_id'])) {
                $req = $req->where('province_id', $dt_filter['province_id']);
            }
            if (!empty($dt_filter['district_id'])) {
                $req = $req->where('district_id', $dt_filter['district_id']);
            }
            // dt_search 
            if (!empty($dt_search)) {
                $req = $req->where(function ($query) use ($dt_search) {
                    $query->whereRelation('local', 'name', 'like', "%$dt_search%")
                        ->orwhere('zipcode', 'like', "%$dt_search%");
                });
            }

            // count all Cities
            $dt_total = $req->count();
            // set query order & limit from datatable
            $resp = $req->orderBy($dt_column[$dt_order], $dt_dir)
                ->offset($dt_start)
                ->limit($dt_length)
                ->get();
            // query Cities

            // prepare datatable for response
            $tables = datatables($resp)
                ->addIndexColumn()
                ->setRowId('id')
                ->setRowClass('city_row')
                ->setTotalRecords($dt_total)
                ->setFilteredRecords($dt_total)
                ->setOffset($dt_start)
                ->editColumn('updated_at', function ($record) {
                    return str_replace(' ', '<br>', date("Y-m-d H:i:s", strtotime($record->updated_at)));
                })->editColumn('name', function ($record) {
                    return $record->local->name ?? '';
                })->editColumn('district_id', function ($record) {
                    return $record->district->local->name ?? '';
                })->editColumn('province_id', function ($record) {
                    return $record->province->local->name ?? '';
                })->addColumn('action', function ($record) {
                    $btn = '<div class="button-list">';
                    $btn .= self::btn_route("admin.mwz.city.edit", $record->id);
                    $btn .= self::btn_delete("admin.mwz.city.set_delete", $record->id, 'setDelete');
                    $btn .= '</div>';
                    return $btn;
                })->escapeColumns([]);
            // response datatable json
            return $tables->make(true);
        }
    }

    /**
     * Function : form
     * Dev : Poom
     * Update Date : 26 Apr 2024
     */
    public function form($id = 0)
    {
        $data = [];
        self::$navbar[] =  ['name' => !empty($id) ? __('action.edit') : __('action.add'), 'url' => null];
        if (!empty($id)) {
            $data = Cities::find($id);
            if (!empty($data->langs)) {
                $data->langs = $data->langs->groupBy('lang')->map(function ($row) {
                    return  $row->first();
                });
            }
        }

        return view('mwz::city.form', ['data' => $data, 'navbar' => self::$navbar]);
    }

    /**
     * Function : save
     * Dev : Poom
     * Update Date : 26 Apr 2024
     */
    public function save(Request $request)
    {
        $rule = [
            "city_id" => 'required',
            "district_id" => 'required',
            "zipcode" => 'required'
        ];
        $msg = [
            "city_id.*" => __('mwz::city.field.city_placeholder'),
            "district_id.*" => __('mwz::city.field.district_placeholder'),
            "zipcode.*" => __('mwz::city.field.zipcode_placeholder')
        ];
        foreach (languages() as $lang => $l) {
            $rule["name_$lang"] = 'required|max:255';
            $msg["name_$lang.*"] = __('mwz::city.field.name_placeholder') . ' ' . ($l['name'] ?? '');
        }

        //validate post data
        $valid = validator($request->all(), $rule, $msg);
        if ($valid->fails()) {
            return self::response(['msg' => $valid->errors()->first(), 'error' => $valid->errors()], 404);
        }
        $attr = [
            'zipcode' => $request->get("zipcode"),
            'latitude' => $request->get("latitude"),
            'longitude' => $request->get("longitude"),
            'province_id' => $request->get("province_id"),
            'district_id' => $request->get("district_id")
        ];
        $set = Cities::updateOrcreate(['id' => $request->get('id')], $attr);
        if ($set->save()) {
            self::setLangs($request, $set->id);
            return self::response(['msg' => __('noti.success')]);
        }

        return self::response(['msg' => __('noti.error')], Response::HTTP_NOT_FOUND);
    }

    /**
     * Function : set translate
     * Dev : Poom
     * Update Date : 26 Apr 2024
     */
    private function setLangs(Request $request, $id)
    {
        foreach (languages() as $lang => $v) {
            CityTranslate::updateOrCreate(
                ['city_id' => $id, 'lang' => $lang],
                ["name" => mwz_setTextString($request->get("name_{$lang}") ?? '')]
            );
        }
    }

    /**
     * Function : set delete
     * Dev : Poom
     * Update Date : 26 Apr 2024
     */
    public function set_delete(Request $request)
    {
        $set = Cities::find($request->get('id'));
        if ($set->delete()) {
            return self::response(['msg' => __('noti.delete')]);
        }

        return self::response(['msg' => __('noti.error')], Response::HTTP_NOT_FOUND);
    }

    /**
     * Function : get list
     * Dev : Poom
     * Update Date : 26 Apr 2024
     */
    public function get_list(Request $request)
    {
        $filter = $request->get('parent');
        $req = new Cities();
        if (!empty($filter['search'])) {
            $req = $req->whereRelation('local', 'name', 'like', "%{$filter['search']}%");
        }
        if (!empty($filter['id'])) {
            $req = $req->where('id', '<>', $filter['id']);
        }
        if (!empty($filter['province_id'])) {
            $req = $req->where('province_id', $filter['province_id']);
        }
        if (!empty($filter['district_id'])) {
            $req = $req->where('district_id', $filter['district_id']);
        }
        $data = $req->orderBy('id', 'asc')->limit(50)->get();
        $result = [];
        foreach ($data as $list) {
            $result[] = ['id' => $list->id, 'text' => $list->local->name ?? null, 'image' =>  ''];
        }

        return self::response(['results' => $result]);
    }

    /**
     * Function : mwz_zipcode
     * Desc: สำหรับหาข้อมูลจังหวัด อำเภอ ตำบล จาก zipcode
     * Dev : Poom
     * Update Date : 26 Apr 2024
     */
    public function mwz_zipcode(Request $request)
    {
        $is_geo = $is_province = $is_district = [];
        $geo = $province = $district = $city = [];
        $o_data = new Cities();
        $o_data = $o_data->where('zipcode', $request->get('zipcode'));
        $list = $o_data->get();
        if (!empty($list)) {
            foreach ($list as $item) {
                $city[] = ['id' => $item->id, 'text' => $item->local->name ?? null, 'geo_id' => $item->geo_id, 'province_id' => $item->province_id, 'district_id' => $item->district_id];
                if (!in_array($item->geo_id, $is_geo)) {
                    $geo[] = ['id' => $item->geo_id, 'text' => $item->geo->local->name ?? null];
                    $is_geo[] = $item->geo_id;
                }
                if (!in_array($item->province_id, $is_province)) {
                    $province[] = ['id' => $item->province_id, 'text' => $item->province->local->name ?? null, 'geo_id' => $item->province->geo_id,];
                    $is_province[] = $item->province_id;
                }
                if (!in_array($item->district_id, $is_district)) {
                    $district[] = ['id' => $item->district_id, 'text' => $item->district->local->name ?? null, 'geo_id' => $item->district->geo_id, 'province_id' => $item->district->province_id,];
                    $is_district[] = $item->district_id;
                }
            }
            $resp = ['geo' => $geo,  'province' => $province,  'district' => $district, 'city' => $city];
        } else {
            $resp = ['msg' => __('noti.no-data')];
        }

        return self::response($resp);
    }

    /**
     * Function : json
     * Desc: สำหรับสร้างข้อมูลจังหวัด อำเภอ ตำบล จากไฟล์ json
     * Dev : Poom
     * Update Date : 26 Apr 2024
     */
    public function json(Request $request)
    {
        $langs = array_keys(languages());
        $index = intval($request->get('index') ?? 1);
        $per_page = 100;
        $offset = ($index - 1) * $per_page;
        $list = array_slice(File::json('assets/json/city.json', JSON_UNESCAPED_UNICODE), $offset, $per_page);
        if (!empty($list)) {
            foreach ($list as $item) {
                if (Cities::where('id', $item['id'])->count() == 0) {
                    $req = Cities::updateOrCreate(
                        ['id' => $item['id']],
                        [
                            'district_id' => $item['district_id'],
                            'zipcode' => $item['zip_code']
                        ]
                    );
                    foreach ($langs as $lang) {
                        $req->langs()->updateOrCreate(
                            ['lang' => $lang],
                            ['name' => $item["name_{$lang}"] ?? $item['name_th'] ?? '']
                        );
                    }
                    $req->geo_id = $req->district->geo_id;
                    $req->province_id = $req->district->province_id;
                    $req->save();
                }
            }
            return self::response(['status' => 0]);
        }
        return self::response(['status' => 1]);
    }
}
