<?php

namespace Modules\Mwz\app\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Route;
use Modules\Mwz\app\Models\Permissions;
use Yajra\DataTables\Facades\DataTables;

class PermissionAdminController extends AdminController
{
	/**
	 * Function : construct
	 * Dev : Tong
	 * Update Date : 29 Apr 2024
	 */
	public function __construct()
	{
		$this->middleware('auth:admin');
		self::$navbar = [
			['name' => __('mwz::permission.title'), 'url' => null],
			['name' => __('mwz::permission.name'), 'url' => route_has('admin.mwz.permission.index')]
		];
	}

	/**
	 * Function : index
	 * Dev : Tong
	 * Update Date : 29 Apr 2024
	 */
	public function index()
	{
		return view('mwz::permission.index', ['navbar' => self::$navbar]);
	}

	/**
	 * Function : datatable ajax
	 * Dev : Tong
	 * Update Date : 29 Apr 2024
	 */
	public function datatable_ajax(Request $request)
	{
		if ($request->ajax()) {
			//init datatable
			$dt_column = array('id', 'name', 'group', 'module', 'page', 'action', 'updated_at');
			$dt_order = $request->get('order')[0]['column'];
			$dt_dir = $request->get('order')[0]['dir'];
			$dt_start = $request->get('start');
			$dt_length = $request->get('length');
			$dt_search = $request->get('search')['value'];
			$dt_filter = $request->get('filter');

			// create permission object
			$req = new Permissions();

			if (!empty($dt_filter['module'])) {
				$req = $req->where('module', $dt_filter['module']);
			}
			if (!empty($dt_filter['page'])) {
				$req = $req->where('page', $dt_filter['page']);
			}
			if (!empty($dt_filter['group'])) {
				$req = $req->where('group', $dt_filter['group']);
			}
			// add search query if have search from datable
			if (!empty($dt_search)) {
				$req = $req->where('name', 'like', "%" . $dt_search . "%");
			}

			$dt_total = $req->count();
			// set query order & limit from datatable
			$resp = $req->orderBy($dt_column[$dt_order], $dt_dir)
				->offset($dt_start)->limit($dt_length)->get();

			// query permission
			// prepare datatable for response
			$tables = DataTables::of($resp)
				->addIndexColumn()
				->setRowId('id')
				->setRowClass('permission_row')
				->setTotalRecords($dt_total)
				->setFilteredRecords($dt_total)
				->setOffset($dt_start)
				->editColumn('updated_at', function ($record) {
					return str_replace(' ', '<br>', date("Y-m-d H:i:s", strtotime($record->updated_at)));
				})->addColumn('manage', function ($record) {
					$btn = '<div class="gap-1 d-flex flex-nowrap">';
					$btn .= self::btn_route('admin.mwz.permission.edit', $record->id);
					$btn .= self::btn_delete('admin.mwz.permission.set_delete', $record->id, 'setDelete');
					$btn .= '</div>';
					return $btn;
				})->escapeColumns([]);

			// response datatable json
			return $tables->make(true);
		}
	}

	/**
	 * Function : form
	 * Dev : Tong
	 * Update Date : 29 Apr 2024
	 */
	public function form($id = 0)
	{
		self::$navbar[] =  ['name' => !empty($id) ? __('action.edit') : __('action.add'), 'url' => null];
		$data = [];
		if (!empty($id)) {
			$data = Permissions::find($id);
		}
		return view('mwz::permission.form', ['data' => $data, 'navbar' => self::$navbar]);
	}

	/**
	 * Function : save permission
	 * Dev : Tong
	 * Update Date : 29 Apr 2024
	 */
	public function save(Request $request)
	{
		//validate post data
		$valid = validator($request->all(), [
			'name' => 'required|max:100',
		], [
			'name.*' => __('mwz::permission.field.name_placeholder'),
		]);

		if ($valid->fails()) {
			return self::response(['msg' => $valid->errors()->first(), 'error' => $valid->errors()], 404);
		}

		$attr['name'] = $request->get('name');

		$set = Permissions::updateOrCreate(['id' => $request->get('id') ?? null], $attr);
		if ($set->save()) {
			return self::response(['msg' => __('noti.success')]);
		}
		return self::response(['msg' => __('noti.error')], 404);
	}

	/**
	 * Function : set status
	 * Dev : Tong
	 * Update Date : 29 Apr 2024
	 */
	public function set_status(Request $request)
	{
		$set = Permissions::find($request->get('id'));
		$set->status = !$set->status;
		if ($set->save()) {
			return self::response(['msg' => __('noti.success')]);
		}
		return self::response(['msg' => __('noti.error')], 404);
	}

	/**
	 * Function : set delete
	 * Dev : Tong
	 * Update Date : 29 Apr 2024
	 */
	public function set_delete(Request $request)
	{
		$set = Permissions::find($request->get('id'));
		if ($set->delete()) {
			return self::response(['msg' => __('noti.delete')]);
		}
		return self::response(['msg' => __('noti.error')], 404);
	}

	/**
	 * Function : get list
	 * Dev : Tong
	 * Update Date : 29 Apr 2024
	 */
	public function get_list(Request $request)
	{
		$filter = $request->get('parent');
		$req = new Permissions();
		if (!empty($filter['search'])) {
			$req = $req->where('name', 'like', "%{$filter['search']}%");
		}
		if (!empty($filter['id'])) {
			$req = $req->where('id', '<>', $filter['id']);
		}
		$req = $req->orderBy('name', 'asc')->get();
		$result = [];
		foreach ($req as $list) {
			$result[] = ['id' => $list->id, 'text' => $list->name, 'image' => ''];
		}
		return self::response(['results' => $result]);
	}

	/**
	 * Function : get route
	 * Dev : Tong
	 * Update Date : 29 Apr 2024
	 */
	public function get_route(Request $request)
	{
		$filter = $request->get('parent');
		$req = new Permissions();
		if (!empty($filter['search'])) {
			$req = $req->where('name', 'like', "%{$filter['search']}%");
		}
		if (!empty($filter['id'])) {
			$req = $req->where('id', '<>', $filter['id']);
		}
		$req = $req->where('action', 'index');
		$req = $req->orderBy('name', 'asc')->get();
		$result = [];
		foreach ($req as $list) {
			$result[] = ['id' => $list->id, 'text' => $list->name,  'image' => ''];
		}
		return self::response(['results' => $result]);
	}

	/**
	 * Function : generate
	 * Dev : Tong
	 * Update Date : 29 Apr 2024
	 */
	public function generate()
	{
		$per = [
			'view' => ['index', 'datatable_ajax', 'get_', 'icons', 'detail', 'view'],
			'add' => ['add', 'save'],
			'edit' => ['edit', 'tree', 'sort', 'tree_sort', 'sitemap'],
			'import' => ['import'],
			'export' => ['export'],
			'delete' => ['set_delete']
		];
		$check = array();
		$collect = Route::getRoutes();
		foreach ($collect as $value) {
			$route_name = $value->getName();
			$route = explode('.', $route_name);
			if (str_contains($route_name, 'admin.') && count($route) == 4) {
				list($admin, $module, $page, $func) = $route;
				if ($module == 'mwz') {
					continue;
				}
				array_push($check, $route_name);

				$group = 'other';
				if (in_array($func, $per['view']) || str_contains($route_name, 'get_')) {
					$group = 'view';
				} elseif (in_array($func, $per['add']) || str_contains($route_name, 'save_')) {
					$group = 'add';
				} elseif (in_array($func, $per['edit']) || str_contains($route_name, 'set_')) {
					$group = 'edit';
				} elseif (in_array($func, $per['import'])) {
					$group = 'import';
				} elseif (in_array($func, $per['export']) || str_contains($route_name, 'export_')) {
					$group = 'export';
				} elseif (in_array($func, $per['delete'])) {
					$group = 'delete';
				}
				// if (empty(Permissions::where('route_name', $route_name)->first())) {
				Permissions::updateOrCreate(
					['route_name' => $route_name],
					[
						'name' => $route_name,
						'group' => $group,
						'module' => $module,
						'role' => str_implode('.', $module, $page),
						'page' => $page,
						'action' => $func,
					]
				);
				// }
			}
		}
		Permissions::whereNotIn('route_name', $check)->delete();
		self::clear_permission();
		return self::response(['msg' => __('noti.success')]);
	}

	/**
	 * Function : clear permission
	 * Dev : Tong
	 * Update Date : 29 Apr 2024
	 */
	public function clear_permission()
	{
		$set = Permissions::all();
		if (!empty($set)) {
			foreach ($set as $item) {
				if (!Route::has($item->route_name)) $item->delete();
			}
		}
	}
}
