<?php

namespace Modules\Mwz\app\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\File;
use Modules\Mwz\app\Models\Provinces;
use Modules\Mwz\app\Models\ProvinceTranslate;

class ProvinceAdminController extends AdminController
{
    /**
     * Function : construct
     * Dev : Poom
     * Update Date : 29 Apr 2024
     */
    public function __construct()
    {
        $this->middleware('auth:admin');
        self::$navbar = [
            ['name' => __("mwz::province.title"), 'url' => null],
            ['name' => __("mwz::province.name"), 'url' => route_has("admin.mwz.province.index")]
        ];
    }

    /**
     * Function : index page
     * Dev : Poom
     * Update Date : 29 Apr 2024
     */
    public function index()
    {
        return view('mwz::province.index', ['navbar' => self::$navbar]);
    }

    /**
     * Function : datatable ajax
     * Dev : Poom
     * Update Date : 29 Apr 2024
     */
    public function datatable_ajax(Request $request)
    {
        if ($request->ajax()) {
            //init datatable
            $dt_column = ['id', 'name', 'geo_id', 'updated_at', 'action'];
            $dt_order = $request->get('order')[0]['column'];
            $dt_dir = $request->get('order')[0]['dir'];
            $dt_start = $request->get('start');
            $dt_length = $request->get('length');
            $dt_search = $request->get('search')['value'];
            $dt_filter = $request->get('filter');

            // create menu cat object
            $req = new Provinces();

            if (!empty($dt_filter['geo_id'])) {
                $req = $req->where('geo_id', $dt_filter['geo_id']);
            }
            // dt_search 
            if (!empty($dt_search)) {
                $req = $req->where(function ($query) use ($dt_search) {
                    $query->whereRelation('local', 'name', 'like', "%$dt_search%");
                });
            }

            // count all menu cat
            $dt_total = $req->count();
            // set query order & limit from datatable
            $resp = $req->orderBy($dt_column[$dt_order], $dt_dir)
                ->offset($dt_start)
                ->limit($dt_length)
                ->get();

            // prepare datatable for response
            $tables = datatables($resp)
                ->addIndexColumn()
                ->setRowId('id')
                ->setRowClass('province_row')
                ->setTotalRecords($dt_total)
                ->setFilteredRecords($dt_total)
                ->setOffset($dt_start)
                ->editColumn('updated_at', function ($record) {
                    return str_replace(' ', '<br>', date("Y-m-d H:i:s", strtotime($record->updated_at)));
                })->editColumn('name', function ($record) {
                    return $record->local->name ?? '';
                })->editColumn('geo_id', function ($record) {
                    return $record->geo->local->name ?? '';
                })->addColumn('action', function ($record) {
                    $btn = '<div class="button-list">';
                    $btn .= self::btn_route("admin.mwz.province.edit", $record->id);
                    $btn .= self::btn_delete("admin.mwz.province.set_delete", $record->id, 'setDelete');
                    $btn .= '</div>';
                    return $btn;
                })->escapeColumns([]);
            // response datatable json
            return $tables->make(true);
        }
    }

    /**
     * Function : form
     * Dev : Poom
     * Update Date : 29 Apr 2024
     */
    public function form($id = 0)
    {
        $data = [];
        self::$navbar[] =  ['name' => !empty($id) ? __('action.edit') : __('action.add'), 'url' => null];
        if (!empty($id)) {
            $data = Provinces::find($id);
            if (!empty($data->langs)) {
                $data->langs = $data->langs->groupBy('lang')->map(function ($row) {
                    return  $row->first();
                });
            }
        }
        return view('mwz::province.form', ['data' => $data, 'navbar' => self::$navbar]);
    }

    /**
     * Function : save 
     * Dev : Poom
     * Update Date : 29 Apr 2024
     */
    public function save(Request $request)
    {
        $rule = $msg = [];
        foreach (languages() as $lang => $l) {
            $rule["name_$lang"] = 'required|max:255';
            $msg["name_$lang.*"] = __('mwz::city.field.name_placeholder') . ' ' . ($l['name'] ?? '');
        }
        $rule["geo_id"] = 'required';
        $msg["geo_id.*"] = __('mwz::province.field.geo_placeholder');
        $valid = validator($request->all(), $rule, $msg);
        if ($valid->fails()) {
            return self::response(['msg' => $valid->errors()->first(), 'error' => $valid->errors()], 303);
        }
        $attr['geo_id'] = $request->get("geo_id");
        $set = Provinces::updateOrcreate(['id' => $request->get('id')], $attr);
        if ($set->save()) {
            self::setLangs($request, $set->id);
            return self::response(['msg' => __('noti.success')]);
        }
        return self::response(['msg' => __('noti.error')], 404);
    }

    /**
     * Function : set translate
     * Dev : Poom
     * Update Date : 29 Apr 2024
     */
    private function setLangs(Request $request, $id)
    {
        foreach (languages() as $lang => $v) {
            ProvinceTranslate::updateOrCreate(
                ['province_id' => $id, 'lang' => $lang],
                ["name" => mwz_setTextString($request->get("name_{$lang}") ?? '')]
            );
        }
    }

    /**
     * Function : delete
     * Dev : Poom
     * Update Date : 29 Apr 2024
     */
    public function set_delete(Request $request)
    {
        $set = Provinces::find($request->get('id'));
        if ($set->delete()) {
            ProvinceTranslate::where('province_id', $request->get('id'))->delete();
            return self::response(['msg' => __('noti.delete')]);
        }
        return self::response(['msg' => __('noti.error')], 404);
    }

    /**
     * Function : get list
     * Dev : Poom
     * Update Date : 29 Apr 2024
     */
    public function get_list(Request $request)
    {
        $filter = $request->get('parent');
        $req = new Provinces();
        if (!empty($filter['search'])) {
            $req = $req->whereRelation('local', 'name', 'like', "%{$filter['search']}%");
        }
        if (!empty($filter['id'])) {
            $req = $req->where('id', '<>', $filter['id']);
        }
        $data = $req->orderBy('id', 'asc')->get();
        $result = [];
        foreach ($data as $list) {
            $result[] = ['id' => $list->id, 'text' => $list->local->name, 'image' =>  ''];
        }
        return self::response(['results' => $result]);
    }

    /**
     * Function : json
     * Dev : Poom
     * Update Date : 29 Apr 2024
     */
    public function json()
    {
        $langs = array_keys(languages());
        Provinces::TRUNCATE();
        ProvinceTranslate::TRUNCATE();
        $mock = File::json('assets/json/provinces.json');
        foreach ($mock as  $item) {
            if (Provinces::where('id', $item['id'])->count() == 0) {
                $req = Provinces::create(['id' => $item['id'], 'geo_id' => $item['geography_id']]);
                foreach ($langs as $lang) {
                    $req->langs()->updateOrCreate(
                        ['lang' => $lang],
                        ['name' => $item["name_{$lang}"] ?? $item['name_th'] ?? '']
                    );
                }
            }
        }
        return self::response(['status' => 1]);
    }
}
