/**
 * Global Language Loader for All Modules
 * Works with Vite build system to ensure $_LANG is available globally
 * Language files are already loaded by Vite, this just initializes them
 */

(function() {
    'use strict';
    
    // Get current locale from meta tag or default to 'en'
    function getCurrentLocale() {
        const metaLocale = document.querySelector('meta[name="locale"]');
        if (metaLocale) {
            return metaLocale.getAttribute('content');
        }
        
        // Fallback to checking HTML lang attribute
        const htmlLang = document.documentElement.lang;
        if (htmlLang) {
            return htmlLang;
        }
        
        // Default fallback
        return 'en';
    }
    
    // Apply jQuery validation messages
    function applyJQueryValidation() {
        if (typeof $.validator !== 'undefined' && window.$_LANG && window.$_LANG.jquery_validate) {
            try {
                $.extend($.validator.messages, window.$_LANG.jquery_validate);
                console.log('jQuery validation messages applied successfully');
            } catch (error) {
                console.warn('Failed to apply jQuery validation messages:', error);
            }
        }
    }
    
    // Wait for language object to be available
    function waitForLanguage() {
        return new Promise((resolve, reject) => {
            // Check if language object is already available
            if (window.$_LANG) {
                console.log('Language object already available');
                resolve();
                return;
            }
            
            // Wait for language object to become available
            let attempts = 0;
            const maxAttempts = 100; // 10 seconds with 100ms intervals
            
            const checkLanguage = setInterval(() => {
                attempts++;
                
                if (window.$_LANG) {
                    clearInterval(checkLanguage);
                    console.log('Language object found after waiting');
                    resolve();
                } else if (attempts >= maxAttempts) {
                    clearInterval(checkLanguage);
                    console.warn('Language object not found after timeout, creating fallback');
                    createFallbackLanguage();
                    resolve();
                }
            }, 100);
        });
    }
    
    // Create fallback language object if needed
    function createFallbackLanguage() {
        window.$_LANG = {
            datatable: {
                lengthMenu: "_MENU_ items",
                zeroRecords: "No records found",
                emptyTable: "No data available",
                info: "_START_-_END_ of _TOTAL_ items",
                infoEmpty: "No items found",
                infoFiltered: "(filtered from _MAX_ total items)",
                loadingRecords: "Loading...",
                processing: "Processing...",
                search: "",
                searchPlaceholder: "Search...",
                paginate: {
                    first: "First",
                    last: "Last",
                    next: "Next",
                    previous: "Previous"
                }
            },
            noti: {
                success: 'Changes saved successfully',
                error: 'An error occurred, please try again',
                warning: 'Warning',
                info: 'Information',
                delete_success: 'Item deleted successfully',
                save_success: 'Data saved successfully',
                update_success: 'Data updated successfully',
                upload_success: 'Upload successful',
                upload_error: 'Upload error occurred',
                file_too_large: 'File is too large',
                invalid_file_type: 'Invalid file type',
                network_error: 'Network connection error',
                server_error: 'Server error occurred',
                validation_error: 'Please check the entered data',
                email_required: 'Please enter email',
                email_invalid: 'Invalid email format',
                email_missing_at: 'Email must contain @',
                email_validation_error: 'Please check email format before submitting form',
            }
        };
    }
    
    // Initialize language system
    async function initLanguage() {
        try {
            const currentLocale = getCurrentLocale();
            console.log(`Current locale: ${currentLocale}`);
            
            // Wait for language object to be available
            await waitForLanguage();
            
            // Try to apply jQuery validation immediately
            applyJQueryValidation();
            
            // Also try to apply when jQuery validation becomes available
            if (typeof $ === 'undefined' || typeof $.validator === 'undefined') {
                // Wait for jQuery and jQuery validation to be available
                const checkJQuery = setInterval(() => {
                    if (typeof $ !== 'undefined' && typeof $.validator !== 'undefined') {
                        clearInterval(checkJQuery);
                        applyJQueryValidation();
                    }
                }, 100);
                
                // Timeout after 10 seconds
                setTimeout(() => {
                    clearInterval(checkJQuery);
                    console.warn('jQuery validation not available after timeout');
                }, 10000);
            }
            
            // Dispatch custom event when language is ready
            window.dispatchEvent(new CustomEvent('languageReady', { detail: { locale: currentLocale } }));
            
        } catch (error) {
            console.error('Failed to initialize language:', error);
            
            // Create fallback language object
            createFallbackLanguage();
            
            // Dispatch custom event even with fallback object
            window.dispatchEvent(new CustomEvent('languageReady', { detail: { locale: 'en', fallback: true } }));
        }
    }
    
    // Run immediately to ensure language is initialized
    initLanguage();
    
    // Export for global access if needed
    window.MwzLanguageLoader = {
        getCurrentLocale: getCurrentLocale,
        waitForLanguage: waitForLanguage,
        initLanguage: initLanguage,
        applyJQueryValidation: applyJQueryValidation
    };
    
})();
