<?php

namespace Modules\Policy\app\Http\Controllers;

use Illuminate\Http\Request;
use Modules\Mwz\app\Http\Controllers\AdminController;
use Modules\Mwz\app\Http\Controllers\SlugController;
use Modules\Policy\app\Models\Cookies;
use Modules\Policy\app\Models\CookiesTranslate;

class CookiesAdminController extends AdminController
{
	private $config;
	/**
	 * Function : construct
	 * Dev : Poom
	 * Update Date : 29 Apr 2024
	 */
	public function __construct()
	{
		$this->middleware('auth:admin');

		self::$navbar = [
			['name' => __("policy::cookies.title"), 'url' => null],
			['name' => __("policy::cookies.main"), 'url' => null],
			['name' => __("policy::cookies.name"), 'url' => route_has("admin.cookies.list.index")]
		];

		$this->config = config("policy.cookies");
	}

	/**
	 * Function : index
	 * Dev : Poom
	 * Update Date : 29 Apr 2024
	 */
	public function index()
	{
		return view('policy::cookies.index', ['navbar' => self::$navbar, 'config' => $this->config]);
	}

	/**
	 * Function : datatable ajax
	 * Dev : Poom
	 * Update Date : 29 Apr 2024
	 */
	public function datatable_ajax(Request $request)
	{
		if ($request->ajax()) {
			//init datatable
			$dt_column = ['sequence', 'code'];
			if ($this->config['input']['name']['status']) {
				array_push($dt_column, 'name');
			}
			array_push($dt_column, 'sort', 'updated_at', 'action');

			$dt_order = $request->get('order')[0]['column'];
			$dt_dir = $request->get('order')[0]['dir'];
			$dt_start = $request->get('start');
			$dt_length = $request->get('length');
			$dt_search = $request->get('search')['value'];
			$dt_filter = $request->get('filter');

			// create Cookies object
			$req = new Cookies();

			// dt_search 
			if (!empty($dt_search)) {
				$req = $req->where(function ($query) use ($dt_search) {
					$query->orWhere('name', 'like', "%$dt_search%");
				});
			}

			// count all Cookies
			$dt_total = $req->count();
			// set query order & limit from datatable
			$resp = $req->orderBy($dt_column[$dt_order], $dt_dir)
				->offset($dt_start)->limit($dt_length)->get();

			// prepare datatable for response
			$tables = datatables($resp)
				->addIndexColumn()
				->setRowId('id')
				->setRowClass('cookie_row')
				->setTotalRecords($dt_total)
				->setFilteredRecords($dt_total)
				->setOffset($dt_start)
				->editColumn('updated_at', function ($record) {
					return str_replace(' ', '<br>', date("Y-m-d H:i:s", strtotime($record->updated_at)));
				})->editColumn('name', function ($record) {
					return $record->local->name ?? '';
				})->editColumn('sort', function ($record) use ($dt_order) {
					return self::btn_sort("admin.cookies.list.set_sort", $record->id, 'setSort', 0, $dt_order != 0);
				})->addColumn('action', function ($record) {
					$btn = '<div class="button-list">';
					$btn .= self::btn_default("admin.cookies.list.set_default", $record->id, 'setDefault', $record->default);
					$btn .= self::btn_status("admin.cookies.list.set_status", $record->id, 'setStatus', $record->status);
					$btn .= self::btn_route("admin.cookies.list.edit", $record->id);
					$btn .= '</div>';
					return $btn;
				})->escapeColumns([]);
			// response datatable json
			return $tables->make(true);
		}
	}

	/**
	 * Function : form
	 * Dev : Poom
	 * Update Date : 29 Apr 2024
	 */
	public function form($id = 0)
	{
		$data = [];
		$meta = $this->set_metadata('policy', 'cookies', 1);
		self::$navbar[] =  ['name' => !empty($id) ? __('action.edit') : __('action.add'), 'url' => null];
		if (!empty($id)) {
			$data = Cookies::find($id);
			if (!empty($data->langs)) {
				$data->langs = $data->langs->groupBy('lang')->map(function ($row) {
					return  $row->first();
				});
			}
			$meta = array_merge($meta, SlugController::getMetadata('policy', 'cookies', $id));
		}
		return view('policy::cookies.form', ['data' => $data, 'config' => $this->config, 'navbar' => self::$navbar, 'meta' => $meta]);
	}

	/**
	 * Function : save
	 * Dev : Poom
	 * Update Date : 29 Apr 2024
	 */
	public function save(Request $request)
	{
		$rule = $msg = [];
		foreach (languages() as $lang => $l) {
			foreach ($this->config['input'] as $key => $item) {
				if (!empty($item['status']) && !empty($item['validate'])) {
					$rule["{$key}_{$lang}"] = ($key != 'name') ? 'required' : 'required|max:255';
					$msg["{$key}_{$lang}.*"] = __("field.{$key}_placeholder") . ' ' . ($l['name'] ?? '');
				}
			}
		}
		//validate post data
		$valid = validator($request->all(), $rule, $msg);
		if ($valid->fails()) {
			return self::response(['msg' => $valid->errors()->first(), 'error' => $valid->errors()], 404);
		}

		$attr['status'] = $request->get('status') ?? 0;

		if (empty($request->get('id')))  $attr["sequence"] = Cookies::max('sequence') + 1;
		$set = Cookies::updateOrcreate(['id' => $request->get('id')], $attr);
		if ($set->save()) {
			self::setLangs($request, $set);
			if ($this->config['seo']) {
				SlugController::createMetadata($request, $set->id);
			}
			return self::response(['msg' => __('noti.success')], 200);
		}
		return self::response(['msg' => __('noti.error')], 404);
	}

	/**
	 * Function : set translate
	 * Dev : Poom
	 * Update Date : 29 Apr 2024
	 */
	private function setLangs(Request $request, $row)
	{
		foreach (array_keys(languages()) as $lang) {
			$attr = [];
			foreach ($this->config['input'] as $key => $item) {
				$attr[$key] = mwz_setTextString($request->get("{$key}_{$lang}") ?? '', $item['type'] == 'texteditor');
			}
			$row->langs()->updateOrCreate(['lang' => $lang], $attr);
		}
	}

	/**
	 * Function : set default
	 * Dev : Poom
	 * Update Date : 29 Apr 2024
	 */
	public function set_default(Request $request)
	{
		$set = Cookies::find($request->get('id'));
		$set->default = !$set->default;
		if ($set->save()) {
			return self::response(['msg' => __('noti.success')]);
		}
		return self::response(['msg' => __('noti.error')], 404);
	}

	/**
	 * Function : set status
	 * Dev : Poom
	 * Update Date : 29 Apr 2024
	 */
	public function set_status(Request $request)
	{
		$set = Cookies::find($request->get('id'));
		$set->status = !$set->status;
		if ($this->config['seo']) {
			SlugController::slug_status('policy', 'cookies', $set->id, $set->status);
		}
		if ($set->save()) {
			return self::response(['msg' => __('noti.success')], 200);
		}
		return self::response(['msg' => __('noti.error')], 404);
	}

	/**
	 * Function : set sort
	 * Dev : Poom
	 * Update Date : 29 Apr 2024
	 */
	public function set_sort(Request $request)
	{
		$node = Cookies::find($request->get('id'));
		$move = $request->get('move') ?? 'up';
		$sort = $move == 'up' ? $node->decrement('sequence') : $node->increment('sequence');
		if ($sort) {
			self::re_order($move == 'up' ? 'DESC' : 'ASC');
			return self::response(['msg' =>  __('noti.success')], 200);
		}
		return self::response(['msg' => __('noti.error')], 404);
	}

	/**
	 * Function : re order
	 * Dev : Poom
	 * Update Date : 29 Apr 2024
	 */
	public function re_order($dir = 'DESC')
	{
		$list = Cookies::orderBy('sequence')->orderBy('updated_at', $dir)->get();
		foreach ($list as $cnt => $cat) {
			$cat->sequence = $cnt + 1;
			$cat->save();
		}
	}
}
