<?php

namespace Modules\Setting\app\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\File;
use Modules\Mwz\app\Http\Controllers\AdminController;
use Modules\Setting\app\Models\Languages;

class LanguagesAdminController extends AdminController
{
    /**
     * Function : construct
     * Dev : Poom
     * Update Date : 29 Apr 2024
     */
    public function __construct()
    {
        $this->middleware('auth:admin');
        self::$navbar = [
            ['name' => __('setting::language.title'), 'url' => null],
            ['name' => __('setting::language.name'), 'url' => route_has('admin.setting.language.index')]
        ];
    }

    /**
     * Function : index
     * Dev : Poom
     * Update Date : 29 Apr 2024
     */
    public function index()
    {
        return view('setting::language.index', ['navbar' => self::$navbar]);
    }

    /**
     * Function : datatable ajax
     * Dev : Poom
     * Update Date : 29 Apr 2024
     */
    public function datatable_ajax(Request $request)
    {
        if ($request->ajax()) {
            //init datatable
            $dt_column = array('code', 'flag', 'name', 'updated_at', 'action');
            $dt_order = $request->get('order')[0]['column'];
            $dt_dir = $request->get('order')[0]['dir'];
            $dt_start = $request->get('start');
            $dt_length = $request->get('length');
            $dt_search = $request->get('search')['value'];

            // create brand object
            $req = new Languages();

            // add search query if have search from datable
            if (!empty($dt_search)) {
                $req = $req->where(
                    function ($q) use ($dt_search) {
                        $q->where('name', 'like', "%$dt_search%")
                            ->orWhere('code', 'like', "%$dt_search%");
                    }
                );
            }

            $dt_total = $req->count();
            // set query order & limit from datatable
            $resp = $req->orderBy($dt_column[$dt_order], $dt_dir)
                ->offset($dt_start)->limit($dt_length)
                ->get();

            // prepare datatable for response
            $tables = datatables($resp)
                ->addIndexColumn()
                ->setRowId('id')
                ->setRowClass('language_row')
                ->setTotalRecords($dt_total)
                ->setFilteredRecords($dt_total)
                ->setOffset($dt_start)
                ->editColumn('code', function ($record) {
                    return  strtoupper($record->code ?? '');
                })->editColumn('flag', function ($record) {
                    return !empty($record->flag) ? ('<i class="' . ($record->flag) . ' fs-3" ></i>') : '';
                })->editColumn('name', function ($record) {
                    return $record->name ?? '';
                })->editColumn('updated_at', function ($record) {
                    return str_replace(' ', '<br>', date("Y-m-d H:i:s", strtotime($record->updated_at)));
                })->addColumn('action', function ($record) {
                    $btn = '<div class="d-flex flex-nowrap gap-1">';
                    $btn .= self::btn_default("admin.setting.language.set_default", $record->id, 'setDefault', $record->default, 'mdi mdi-pin');
                    $btn .= self::btn_status('admin.setting.language.set_status', $record->id, 'setStatus', $record->status);
                    $btn .= self::btn_route('admin.setting.language.edit', $record->id);
                    $btn .= '</div>';

                    return $btn;
                })
                ->escapeColumns([]);

            // response datatable json
            return $tables->make(true);
        }
    }

    /**
     * Function : form
     * Dev : Poom
     * Update Date : 29 Apr 2024
     */
    public function form($id = 0)
    {
        self::$navbar[] = ['name' => !empty($id) ? __('action.edit') : __('action.add'), 'url' => null];
        $data = [];
        if (!empty($id)) {
            $data = Languages::find($id);
        }
        return view('setting::language.form', ['data' => $data, 'navbar' => self::$navbar]);
    }

    /**
     * Function : save
     * Dev : Poom
     * Update Date : 29 Apr 2024
     */
    public function save(Request $request)
    {
        $valid = validator($request->all(), [
            'name' => 'required',
            'code' => 'required',
        ], [
            'name.required' => __('field.name_placeholder'),
            'code.required' => __('field.code_placeholder'),
        ]);

        if ($valid->fails()) {
            return self::response(['msg' => $valid->errors()->first(), 'error' => $valid->errors()], 404);
        }
        $set = Languages::updateOrcreate(
            ['id' => $request->get('id')],
            [
                'name' => $request->get('name'),
                'code' => $request->get('code'),
                'flag' => "flag flag-{$request->get('code')}",
                'status' => $request->get('status') ?? 0
            ]
        );
        if ($set->save()) {
            self::gen_language();
            return self::response(['msg' => __('noti.success')]);
        }
        return self::response(['msg' => __('noti.error')], 404);
    }

    /**
     * Function : set status
     * Dev : Poom
     * Update Date : 29 Apr 2024
     */
    public function set_status(Request $request)
    {
        $set = Languages::find($request->get('id'));
        $set->status = !$set->status;
        if ($set->save()) {
            self::gen_language();
            return self::response(['msg' => __('noti.success')]);
        }
        return self::response(['msg' => __('noti.error')], 404);
    }

    /**
     * Function : set default
     * Dev : Poom
     * Update Date : 29 Apr 2024
     */
    public function set_default(Request $request)
    {
        Languages::where('default', 1)->update(['default' => 0]);
        $set = Languages::find($request->get('id'));
        $set->default = 1;
        if ($set->save()) {
            self::gen_language();
            return self::response(['msg' => __('noti.success')]);
        }
        return self::response(['msg' => __('noti.error')], 404);
    }

    /**
     * Function : delete
     * Dev : Poom
     * Update Date : 29 Apr 2024
     */
    public function set_delete(Request $request)
    {
        $set = Languages::find($request->get('id'));
        if ($set->delete()) {
            self::gen_language();
            return self::response(['msg' => __('noti.delete')]);
        }
        return self::response(['msg' => __('noti.error')], 404);
    }


    /**
     * Function : get code
     * Dev : Poom
     * Update Date : 29 Apr 2024
     */
    public function get_code(Request $request)
    {
        $filter = $request->get('parent');
        // get langs
        $req = File::json("assets/json/lang.json");
        if (!empty($filter['search'])) {
            $req = array_filter($req, function ($value) use ($filter) {
                return str_contains($value, strtolower($filter['search']));
            });
        }
        $result = [];
        foreach ($req as $v) {
            $result[] = ['id' => $v, 'text' => $v, 'flag' =>  "flag flag-{$v}"];
        }
        return self::response(['results' => $result]);
    }

    /**
     * Function : get list
     * Dev : Poom
     * Update Date : 29 Apr 2024
     */
    public function get_list(Request $request)
    {
        $filter = $request->get('parent');
        $req = new Languages();
        if (!empty($filter['search'])) {
            $req = $req->where('name', 'like', "%{$filter['search']}%");
        }
        if (!empty($filter['id'])) {
            $req = $req->where('id', '<>', $filter['id']);
        }
        $req = $req->orderBy('sequence')->get();
        $result = [];
        foreach ($req as $list) {
            $result[] = ['id' => $list->code, 'text' => $list->name, 'flag' => $list->flag ?? ''];
        }
        return self::response(['results' => $result]);
    }
    /**
     * Function : gen laguage
     * Dev : Poom
     * Update Date : 29 Apr 2024
     */
    private function gen_language()
    {
        $res = [];
        $req = Languages::where('status', 1)->orderby('default', 'desc')->orderby('code')->get();
        if (!empty($req)) {
            foreach ($req as $item) {
                $res[$item->code] = ['flag' => $item->flag, 'name' => $item->name];
            }
        } else {
            $lang = app()->getLocale() ?? 'th';
            $res[$lang] = ['flag' => "flag flag-{$lang}", 'name' => __("field.lang.$lang")];
        }
        File::put('assets/json/languages.json', json_encode($res, JSON_UNESCAPED_UNICODE));
    }
}
