<?php

namespace Modules\User\app\Http\Controllers;

use Illuminate\Http\Response;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Route;
use Modules\Mwz\app\Http\Controllers\AdminController;
use Modules\Mwz\app\Models\AdminMenus;
use Modules\User\app\Models\Roles;
use Modules\User\app\Models\RolesPermissions;
use Modules\Mwz\app\Models\Permissions as MwzPermissions;

class RoleAdminController extends AdminController
{
	private $action;

	/**
	 * Function : construct
	 * Dev : Petch
	 * Update Date : 22 Jan 2025
	 */
	public function __construct()
	{
		$exp = explode('.', Route::currentRouteName());
		$this->action = $exp[3];

		self::$navbar = [
			['name' => __('user::role.title'), 'url' => null],
			['name' => __('user::role.name'), 'url' => route_has('admin.user.role.index')]
		];
	}

	/**
	 * Function : index
	 * Dev : Poom
	 * Update Date : 29 Apr 2024
	 */
	public function index()
	{
		return view('user::role.index', ['navbar' => self::$navbar]);
	}

	/**
	 * Function : datatable ajax
	 * Dev : Poom
	 * Update Date : 29 Apr 2024
	 */
	public function datatable_ajax(Request $request)
	{
		if ($request->ajax()) {
			//init datatable
			$dt_column = array('sequence', 'name', 'sort', 'updated_at', 'action');
			$dt_order = $request->get('order')[0]['column'];
			$dt_dir = $request->get('order')[0]['dir'];
			$dt_start = $request->get('start');
			$dt_length = $request->get('length');
			$dt_search = $request->get('search')['value'];

			// create role object
			$req = new Roles();

			// add search query if have search from datable
			if (!empty($dt_search)) {
				$req = $req->where('name', 'like', "%" . $dt_search . "%");
			}

			$dt_total = $req->count();
			// set query order & limit from datatable
			$resp = $req->orderBy($dt_column[$dt_order], $dt_dir)->offset($dt_start)
				->limit($dt_length)
				->get();

			// prepare datatable for response
			$tables = datatables($resp)
				->addIndexColumn()
				->setRowId('id')
				->setRowClass('role_row')
				->setTotalRecords($dt_total)
				->setFilteredRecords($dt_total)
				->setOffset($dt_start)
				->editColumn('updated_at', function ($record) {
					return str_replace(' ', '<br>', date("Y-m-d H:i:s", strtotime($record->updated_at)));
				})->editColumn('sort', function ($record) use ($dt_order) {
					return self::btn_sort("admin.user.role.set_sort", $record->id, 'setSort', 0, $dt_order != 0);
				})->editColumn('action', function ($record) {
					$btn = '<div class="button-list">';
					if ($record->id != 1) {
						$btn .= self::btn_status('admin.user.role.set_status', $record->id, 'setStatus', $record->status);
					}
					$btn .= self::btn_route('admin.user.role.edit', $record->id);
					if ($record->id != 1) {
						$btn .= self::btn_delete('admin.user.role.set_delete', $record->id, 'setDelete');
					}
					$btn .= '</div>';
					return $btn;
				})
				->escapeColumns([]);

			// response datatable json
			return $tables->make(true);
		}
	}

	/**
	 * Function : form
	 * Dev : Poom
	 * Update Date : 29 Apr 2024
	 */
	public function form($id = 0)
	{
		self::$navbar[] =  ['name' => __("action.{$this->action}"), 'url' => null];
		$checked = [];
		$oilprice_abc_checked = [];
		if (!empty($id)) {
			$data = Roles::find($id);
			if (!empty($data->permissions)) {
				foreach ($data->permissions as $p)
					$checked[] = $p->id;
			}
			// Load OilPrice ABC permissions from relation (status = 1)
			$oilprice_abc_checked = $data->oilprice_abc()
				->where('status', 1)
				->pluck('name')
				->toArray();
		}
		return view('user::role.form', [
			'data' => $data ?? null,
			'super' => !empty($data) && $data->id == 1,
			'permissions' => self::permission(),
			'checked' => $checked,
			'oilprice_abc_checked' => $oilprice_abc_checked,
			'navbar' => self::$navbar,
			'action' => $this->action
		]);
	}

	/**
	 * Function : persmission
	 * Dev : Wan
	 * Update Date : 20 Jan 2025
	 */
	private static function permission()
	{
		$list = [];
		$menus = AdminMenus::where('status', 1)
			->whereNull('parent_id')
			->with(['children' => function ($q) {
				$q->where('status', 1)->orderBy('_lft');
			}, 'children.children' => function ($q) {
				$q->where('status', 1)->orderBy('_lft');
			}])
			->orderBy('_lft')
			->get();

		if (!empty($menus)) {
			foreach ($menus as $m) {
				if (empty($m->status))
					continue;

				$list[$m->id]['name'] = $m->name ?? '';
				if (!empty($m->permission->roles)) {
					foreach ($m->permission->roles as $p)
						$list[$m->id]['permission'][$p->group][] = $p->id;
				}
				self::child_permission($list[$m->id]['sub_menu'], $m->children);
			}
		}
		return $list;
	}

	/**
	 * Function : child persmission
	 * Dev : Petch
	 * Update Date : 20 Jan 2025
	 */
	private static function child_permission(&$list, $children)
	{
		if (empty($children))
			return;

		foreach ($children as $c) {
			if (empty($c->status))
				continue;

			$list[$c->id]['name'] = $c->name ?? '';
			if (!empty($c->permission->roles)) {
				foreach ($c->permission->roles as $p)
					$list[$c->id]['permission'][$p->group][] = $p->id;
			}
			if (!empty($c->children))
				self::child_permission($list[$c->id]['sub_menu'], $c->children);
		}
	}

	/**
	 * Function : role save
	 * Dev : Petch
	 * Update Date : 21 Jan 2025
	 */
	public function save(Request $request)
	{
		$valid = validator($request->all(), [
			'name' => 'required|max:255',
		], [
			'name.required' => __('field.name_placeholder'),
			'name.max' => __('field.name_max', ['max' => 255]),
		]);

		if ($valid->fails())
			return self::response(['msg' => $valid->errors()->first(), 'error' => $valid->errors()], Response::HTTP_BAD_REQUEST);

		$attr = [
			"name" => $request->get('name'),
			"status" => $request->get('status') ?? 0,
		];

		if (empty($request->get('id')))
			$attr["sequence"] = Roles::max('sequence') + 1;
		$roles = Roles::updateOrCreate(['id' => $request->get('id')], $attr);

		if ($roles->save()) {
			$submittedPermissions = $request->get('permissions');
			// Ensure super role (id = 1) always has all permissions
			$roles->list()->sync($submittedPermissions ?? []);
			
			// Handle OilPrice ABC permissions based on checkboxes
			$oilprice_abc = $request->get('oilprice_abc', []);
			$roles->oilprice_abc()->delete();
			$allAbcKeys = ['value_a', 'value_b', 'value_c'];
			foreach ($allAbcKeys as $abcKey) {
				$roles->oilprice_abc()->create([
					'name' => $abcKey,
					'status' => in_array($abcKey, $oilprice_abc) ? 1 : 0,
				]);
			}
			
			self::re_order();

			return self::response(['msg' => __('noti.success')]);
		}

		return self::response(['msg' => __('noti.error')], Response::HTTP_INTERNAL_SERVER_ERROR);
	}

	/**
	 * Function : set status
	 * Dev : Poom
	 * Update Date : 29 Apr 2024
	 */
	public function set_status(Request $request)
	{
		$role = Roles::find($request->get('id'));
		$role->status = !$role->status;
		if ($role->save()) {
			return self::response(['msg' => __('noti.success')], 200);
		}
		return self::response(['msg' => __('noti.error')], 404);
	}

	/**
	 * Function : set delete
	 * Dev : Poom
	 * Update Date : 29 Apr 2024
	 */
	public function set_delete(Request $request)
	{
		$role = Roles::find($request->get('id'));
		if ($role->delete()) {
			if (RolesPermissions::where('role_id', $request->get('id'))->count()) {
				RolesPermissions::where('role_id', $request->get('id'))->delete();
			}
			self::re_order();
			return self::response(['msg' => __('noti.delete')], 200);
		}
		return self::response(['msg' => __('noti.error')], 404);
	}

	/**
	 * Function : re order
	 * Dev : Poom
	 * Update Date : 29 Apr 2024
	 */
	public function re_order($dir = 'DESC')
	{
		$lists = Roles::orderBy('sequence')->orderBy('updated_at', $dir)->get();
		if (!empty($lists)) {
			foreach ($lists as $k => $row) {
				$row->sequence = $k + 1;
				$row->save();
			}
		}
	}

	/**
	 * Function : set sort
	 * Dev : Poom
	 * Update Date : 29 Apr 2024
	 */
	public function set_sort(Request $request)
	{
		$node = Roles::find($request->get('id'));
		$move = $request->get('move') ?? 'up';
		$sort = $move == 'up' ? $node->decrement('sequence') : $node->increment('sequence');
		if ($sort) {
			self::re_order($move == 'up' ? 'DESC' : 'ASC');
			return self::response(['msg' =>  __('noti.success')], 200);
		}
		return self::response(['msg' => __('noti.error_sort')], 404);
	}
	/**
	 * Function : get list
	 * Dev : Poom
	 * Update Date : 29 Apr 2024
	 */
	public function get_list(Request $request)
	{
		$filter = $request->get('parent');
		$req = Roles::whereNot('id', 1)->where('status', 1);
		if (!empty($filter['search']))
			$req = $req->where('name', 'like', "%{$filter['search']}%");

		if (!empty($filter['id']))
			$req = $req->where('id', '<>', $filter['id']);

		$data = $req->orderBy('sequence')->get();
		$result = [];
		foreach ($data as $list)
			$result[] = ['id' => $list->id, 'text' => $list->name, 'image' => $list->image ?? ''];

		return self::response(['results' => $result]);
	}
}
