/**
 * Record Data - Check-in/Check-out System with Google Distance Matrix API
 */

let currentCheckin = null;
let pendingCheckoutData = null;
let distanceMatrixService = null;

/**
 * Utility Functions
 */
function toDatetimeLocal(date) {
  const pad = n => n.toString().padStart(2, "0");
  return `${date.getFullYear()}-${pad(date.getMonth() + 1)}-${pad(date.getDate())}T${pad(date.getHours())}:${pad(date.getMinutes())}`;
}

function showElement(el) {
  el.classList.remove("hidden");
}

function hideElement(el) {
  el.classList.add("hidden");
}

function clearErrors() {
  document.querySelectorAll(".error-msg").forEach(el => el.textContent = "");
}

function resetToInitialStep() {
  currentCheckin = null;
  hideElement(document.getElementById("checkinForm"));
  hideElement(document.getElementById("checkoutForm"));
  hideElement(document.getElementById("checkoutPreview"));
  hideElement(document.getElementById("checkoutEditForm"));
  hideElement(document.getElementById("latestCheckinBox"));
  hideElement(document.getElementById("btnCheckout"));
  showElement(document.getElementById("btnCheckin"));
  document.querySelectorAll('#checkinForm input,#checkinForm textarea').forEach(el => el.value = "");
  document.querySelectorAll('#checkoutForm input,#checkoutForm textarea').forEach(el => el.value = "");
  document.querySelectorAll('#checkoutEditForm input').forEach(el => el.value = "");
}

/**
 * Haversine formula - Calculate distance between two coordinates
 */
function getDistanceKm(lat1, lon1, lat2, lon2) {
  const R = 6371; // Earth radius in km
  const dLat = (lat2 - lat1) * Math.PI / 180;
  const dLon = (lon2 - lon1) * Math.PI / 180;
  const a = Math.sin(dLat / 2) ** 2 + Math.cos(lat1 * Math.PI / 180) * Math.cos(lat2 * Math.PI / 180) * Math.sin(dLon / 2) ** 2;
  const c = 2 * Math.atan2(Math.sqrt(a), Math.sqrt(1 - a));
  return R * c;
}

/**
 * Google Distance Matrix API - Calculate distance
 */
function calculateDistance(originLat, originLng, destLat, destLng, callback) {
  if (!distanceMatrixService) {
    distanceMatrixService = new google.maps.DistanceMatrixService();
  }

  const origin = new google.maps.LatLng(originLat, originLng);
  const destination = new google.maps.LatLng(destLat, destLng);

  distanceMatrixService.getDistanceMatrix({
    origins: [origin],
    destinations: [destination],
    travelMode: google.maps.TravelMode.DRIVING,
    unitSystem: google.maps.UnitSystem.METRIC,
  }, (response, status) => {
    if (status === 'OK' && response.rows[0].elements[0].status === 'OK') {
      const element = response.rows[0].elements[0];
      callback({
        success: true,
        distance: element.distance.text,
        duration: element.duration.text,
        distanceValue: element.distance.value,
        durationValue: element.duration.value
      });
    } else {
      callback({
        success: false,
        error: status === 'OK' ? 'ไม่พบเส้นทาง' : 'เกิดข้อผิดพลาด: ' + status
      });
    }
  });
}

/**
 * Show distance summary modal
 */
function showDistanceSummaryModal(checkoutData) {
  // Show modal
  showElement(document.getElementById("confirmCheckoutModal"));

  // Show loading state
  showElement(document.getElementById("distanceLoading"));
  hideElement(document.getElementById("distanceSummary"));
  hideElement(document.getElementById("distanceError"));

  // Calculate distance using API
  calculateDistance(
    checkoutData.lat,
    checkoutData.lng,
    checkoutData.targetLat,
    checkoutData.targetLng,
    (result) => {
      hideElement(document.getElementById("distanceLoading"));

      if (result.success) {
        // Show results
        document.getElementById("summaryOrigin").textContent = checkoutData.currentPlace;
        document.getElementById("summaryDestination").textContent = checkoutData.targetPlace;
        document.getElementById("summaryDistance").textContent = result.distance;
        document.getElementById("summaryDuration").textContent = result.duration;
        document.getElementById("summaryCheckinTime").textContent = new Date(checkoutData.checkinTime).toLocaleString('th-TH');
        document.getElementById("summaryCheckoutTime").textContent = new Date(checkoutData.checkoutTime).toLocaleString('th-TH');

        showElement(document.getElementById("distanceSummary"));
      } else {
        // Show error
        document.getElementById("distanceErrorMessage").textContent = result.error;
        showElement(document.getElementById("distanceError"));
      }
    }
  );
}

/**
 * Update preview distance
 */
function updatePreviewDistance() {
  const lat = currentCheckin.lat;
  const lng = currentCheckin.lng;
  const targetLat = currentCheckin.targetLat;
  const targetLng = currentCheckin.targetLng;

  if (targetLat && targetLng) {
    const distance = getDistanceKm(lat, lng, targetLat, targetLng).toFixed(2);
    document.getElementById("previewDistance").textContent = distance;
    document.getElementById("editDistance").textContent = distance;
  } else {
    document.getElementById("previewDistance").textContent = "-";
    document.getElementById("editDistance").textContent = "-";
  }
}

/**
 * Get current location with geocoding
 */
function getCurrentLocation() {
  if (navigator.geolocation) {
    navigator.geolocation.getCurrentPosition(pos => {
      const lat = pos.coords.latitude;
      const lng = pos.coords.longitude;
      const latInput = document.getElementById("lat");
      const lngInput = document.getElementById("lng");
      
      latInput.value = lat.toFixed(6);
      lngInput.value = lng.toFixed(6);
      
      // Dispatch input event to notify Livewire
      latInput.dispatchEvent(new Event('input', { bubbles: true }));
      lngInput.dispatchEvent(new Event('input', { bubbles: true }));

      const geocoder = new google.maps.Geocoder();
      geocoder.geocode({ location: { lat, lng } }, (results, status) => {
        if (status === "OK" && results[0]) {
          const address = results[0].formatted_address;
          const addressInput = document.getElementById("address");
          addressInput.value = address;
          addressInput.dispatchEvent(new Event('input', { bubbles: true }));

          // Get place name from place_id
          const placeId = results[0].place_id;
          if (placeId) {
            const service = new google.maps.places.PlacesService(document.createElement('div'));
            service.getDetails({ placeId: placeId, fields: ['name'] }, (place, status2) => {
              const placeNameInput = document.getElementById("placeName");
              if (status2 === google.maps.places.PlacesServiceStatus.OK && place.name) {
                placeNameInput.value = place.name;
              } else {
                placeNameInput.value = "ไม่พบชื่อสถานที่";
              }
              placeNameInput.dispatchEvent(new Event('input', { bubbles: true }));
            });
          } else {
            const placeNameInput = document.getElementById("placeName");
            placeNameInput.value = "ไม่พบข้อมูล place_id";
            placeNameInput.dispatchEvent(new Event('input', { bubbles: true }));
          }
        } else {
          const addressInput = document.getElementById("address");
          const placeNameInput = document.getElementById("placeName");
          addressInput.value = "";
          placeNameInput.value = "ดึงข้อมูลไม่สำเร็จ";
          addressInput.dispatchEvent(new Event('input', { bubbles: true }));
          placeNameInput.dispatchEvent(new Event('input', { bubbles: true }));
          document.getElementById("locationModalMessage").textContent =
            "เกิดข้อผิดพลาดในการดึงที่อยู่: " + status;
          showElement(document.getElementById("locationModal"));
        }
      });
    }, err => {
      document.getElementById("address").value = "";
      document.getElementById("locationModalMessage").textContent =
        "ไม่สามารถดึงตำแหน่งได้: " + err.message;
      showElement(document.getElementById("locationModal"));
    });
  } else {
    document.getElementById("address").value = "";
    document.getElementById("locationModalMessage").textContent =
      "เบราว์เซอร์ของคุณไม่รองรับ Geolocation";
    showElement(document.getElementById("locationModal"));
  }
}

/**
 * Google Maps Autocomplete
 */
function initAutocomplete() {
  // Autocomplete for place name
  new google.maps.places.Autocomplete(document.getElementById("placeName"));
}

// Make initAutocomplete available globally for Google Maps callback
window.initAutocomplete = initAutocomplete;

/**
 * Save Checkout Data
 */
function saveCheckoutData(data) {
  clearErrors();
  const { currentPlace, targetPlace, reason, checkinTime, checkoutTime, lat, lng, targetLat, targetLng } = data;
  let hasError = false;

  if (!currentPlace) {
    document.getElementById("error-currentPlaceOut").textContent = "กรอกชื่อสถานที่ปัจจุบัน";
    hasError = true;
  }
  if (!targetPlace) {
    document.getElementById("error-targetPlaceOut").textContent = "กรอกชื่อสถานที่จะไป";
    hasError = true;
  }
  if (!reason) {
    document.getElementById("error-reasonOut").textContent = "กรอกเหตุผล";
    hasError = true;
  }
  if (!checkoutTime) {
    document.getElementById("error-checkoutTimeOut").textContent = "เลือกวันที่และเวลา";
    hasError = true;
  }
  if (hasError) return;

  let distance = '-';
  if (lat && lng && targetLat && targetLng) {
    distance = getDistanceKm(lat, lng, targetLat, targetLng).toFixed(2);
  }

  const tbody = document.getElementById("history-body");
  const tr = document.createElement("tr");
  tr.innerHTML = `<td>${new Date(checkoutTime).toLocaleDateString()}</td>
                <td>${new Date(checkinTime).toLocaleTimeString()}</td>
                <td>${new Date(checkoutTime).toLocaleTimeString()}</td>
                <td>${currentPlace} → ${targetPlace}</td>
                <td>${distance}</td>`;
  tbody.appendChild(tr);
  showElement(document.getElementById("checkoutModal"));
}

/**
 * Event Listeners - Initialize when DOM is ready
 */
document.addEventListener('DOMContentLoaded', function () {

  // Check-in button
  document.getElementById("btnCheckin").addEventListener("click", () => {
    showElement(document.getElementById("checkinForm"));
    getCurrentLocation();
    document.getElementById("checkinTime").value = toDatetimeLocal(new Date());
  });

  // Save Check-in
  document.getElementById("saveCheckin").addEventListener("click", () => {
    clearErrors();
    const currentPlace = document.getElementById("currentPlace").value.trim();
    const targetPlace = document.getElementById("targetPlace").value.trim();
    const reason = document.getElementById("reason").value.trim();
    const placeName = document.getElementById("placeName").value.trim();
    const lat = parseFloat(document.getElementById("lat").value.trim());
    const lng = parseFloat(document.getElementById("lng").value.trim());
    const address = document.getElementById("address").value.trim();
    const time = document.getElementById("checkinTime").value;
    let hasError = false;

    if (!currentPlace) {
      document.getElementById("error-currentPlace").textContent = "ใส่ชื่อสถานที่ปัจจุบัน";
      hasError = true;
    }
    if (!targetPlace) {
      document.getElementById("error-targetPlace").textContent = "ใส่ชื่อสถานที่จะไป";
      hasError = true;
    }
    if (!reason) {
      document.getElementById("error-reason").textContent = "ใส่เหตุผล";
      hasError = true;
    }
    if (!time) {
      document.getElementById("error-checkinTime").textContent = "เลือกวันที่และเวลา";
      hasError = true;
    }
    if (hasError) return;

    // Store Check-in data (no destination coordinates yet)
    currentCheckin = { currentPlace, targetPlace, reason, placeName, lat, lng, address, time, targetLat: null, targetLng: null };

    document.getElementById("latestLat").textContent = currentCheckin.lat;
    document.getElementById("latestLng").textContent = currentCheckin.lng;
    showElement(document.getElementById("latestCheckinBox"));

    // Prepare data for Check-out
    document.getElementById("previewCurrentPlace").textContent = currentCheckin.currentPlace;
    document.getElementById("previewTargetPlace").textContent = currentCheckin.targetPlace;
    document.getElementById("previewReason").textContent = currentCheckin.reason;
    document.getElementById("previewCheckinTime").textContent = new Date(currentCheckin.time).toLocaleString();
    document.getElementById("checkinTimeOutPreview").value = toDatetimeLocal(new Date());

    updatePreviewDistance();

    // Hide Check-in form and show Check-out button
    hideElement(document.getElementById("checkinForm"));
    hideElement(document.getElementById("btnCheckin"));
    showElement(document.getElementById("btnCheckout"));
    hideElement(document.getElementById("checkoutForm"));
    hideElement(document.getElementById("checkoutPreview"));
    hideElement(document.getElementById("checkoutEditForm"));

    alert("✅ บันทึก Check-in สำเร็จ!\n\nเมื่อคุณถึงปลายทาง กรุณากดปุ่ม Check-out");
  });

  // Check-out button - Get current location and show form
  document.getElementById("btnCheckout").addEventListener("click", () => {
    if (!currentCheckin) {
      alert("⚠️ กรุณา Check-in ก่อน");
      return;
    }

    // Get current location (destination)
    if (navigator.geolocation) {
      navigator.geolocation.getCurrentPosition(pos => {
        const targetLat = pos.coords.latitude;
        const targetLng = pos.coords.longitude;

        // Store destination coordinates
        currentCheckin.targetLat = targetLat;
        currentCheckin.targetLng = targetLng;

        // Update distance
        updatePreviewDistance();

        // Show Check-out form
        hideElement(document.getElementById("btnCheckout"));
        showElement(document.getElementById("checkoutForm"));
        showElement(document.getElementById("checkoutPreview"));
        hideElement(document.getElementById("checkoutEditForm"));
      }, err => {
        document.getElementById("locationModalMessage").textContent = "ไม่สามารถดึงตำแหน่งปัจจุบันได้: " + err.message;
        showElement(document.getElementById("locationModal"));
      });
    } else {
      alert("⚠️ เบราว์เซอร์ของคุณไม่รองรับ Geolocation");
    }
  });

  // Edit Checkout
  document.getElementById("editCheckout").addEventListener("click", () => {
    hideElement(document.getElementById("checkoutPreview"));
    showElement(document.getElementById("checkoutEditForm"));
    document.getElementById("currentPlaceOut").value = currentCheckin.currentPlace;
    document.getElementById("targetPlaceOut").value = currentCheckin.targetPlace;
    document.getElementById("reasonOut").value = currentCheckin.reason;
    document.getElementById("checkinTimeOut").textContent = new Date(currentCheckin.time).toLocaleString();
    document.getElementById("checkoutTimeOut").value = toDatetimeLocal(new Date());
    updatePreviewDistance();
  });

  // Save Preview Checkout - Show distance summary modal
  document.getElementById("saveCheckoutPreview").addEventListener("click", () => {
    const checkoutTime = document.getElementById("checkinTimeOutPreview").value;
    if (!checkoutTime) {
      document.getElementById("error-checkinTimeOutPreview").textContent = "เลือกวันที่และเวลา";
      return;
    }
    clearErrors();

    // Prepare data
    const checkoutData = {
      currentPlace: currentCheckin.currentPlace,
      targetPlace: currentCheckin.targetPlace,
      reason: currentCheckin.reason,
      checkinTime: currentCheckin.time,
      checkoutTime: checkoutTime,
      lat: currentCheckin.lat,
      lng: currentCheckin.lng,
      targetLat: currentCheckin.targetLat,
      targetLng: currentCheckin.targetLng
    };

    // Store data for saving
    pendingCheckoutData = checkoutData;

    // Show distance summary modal
    showDistanceSummaryModal(checkoutData);
  });

  // Confirm save in distance summary modal
  document.getElementById("confirmSaveCheckout").addEventListener("click", () => {
    hideElement(document.getElementById("confirmCheckoutModal"));

    if (pendingCheckoutData) {
      saveCheckoutData(pendingCheckoutData);
      pendingCheckoutData = null;
    }
  });

  // Cancel buttons in modal
  ["closeConfirmModal", "cancelConfirmCheckout"].forEach(id => {
    document.getElementById(id).addEventListener("click", () => {
      hideElement(document.getElementById("confirmCheckoutModal"));
      pendingCheckoutData = null;
    });
  });

  // Close error button
  document.getElementById("closeDistanceError").addEventListener("click", () => {
    hideElement(document.getElementById("confirmCheckoutModal"));
    pendingCheckoutData = null;
  });

  // Retry button
  document.getElementById("retryDistance").addEventListener("click", () => {
    if (pendingCheckoutData) {
      showDistanceSummaryModal(pendingCheckoutData);
    }
  });

  // Save Edit Checkout - Show distance summary modal
  document.getElementById("saveCheckoutEdit").addEventListener("click", () => {
    clearErrors();
    const currentPlace = document.getElementById("currentPlaceOut").value.trim();
    const targetPlace = document.getElementById("targetPlaceOut").value.trim();
    const reason = document.getElementById("reasonOut").value.trim();
    const checkinTime = currentCheckin.time;
    const checkoutTime = document.getElementById("checkoutTimeOut").value;

    // Validate data
    let hasError = false;
    if (!currentPlace) {
      document.getElementById("error-currentPlaceOut").textContent = "กรอกชื่อสถานที่ปัจจุบัน";
      hasError = true;
    }
    if (!targetPlace) {
      document.getElementById("error-targetPlaceOut").textContent = "กรอกชื่อสถานที่จะไป";
      hasError = true;
    }
    if (!reason) {
      document.getElementById("error-reasonOut").textContent = "กรอกเหตุผล";
      hasError = true;
    }
    if (!checkoutTime) {
      document.getElementById("error-checkoutTimeOut").textContent = "เลือกวันที่และเวลา";
      hasError = true;
    }
    if (hasError) return;

    // Prepare data
    const checkoutData = {
      currentPlace, targetPlace, reason, checkinTime, checkoutTime,
      lat: currentCheckin.lat,
      lng: currentCheckin.lng,
      targetLat: currentCheckin.targetLat,
      targetLng: currentCheckin.targetLng
    };

    // Store data for saving
    pendingCheckoutData = checkoutData;

    // Show distance summary modal
    showDistanceSummaryModal(checkoutData);
  });

  // Modal close buttons
  ["closeModal", "modalConfirm"].forEach(id => {
    document.getElementById(id).addEventListener("click", () => {
      hideElement(document.getElementById("checkoutModal"));
      resetToInitialStep();
    });
  });

  ["closeLocationModal", "locationModalOk"].forEach(id => {
    document.getElementById(id).addEventListener("click", () => {
      hideElement(document.getElementById("locationModal"));
    });
  });

});

