import { test, expect } from '@playwright/test';

// Configure tests to run sequentially to avoid session conflicts
test.describe.configure({ mode: 'serial' });

test.describe('Admin Policy Privacy Page Tests', () => {
  test.beforeEach(async ({ page }) => {
    try {
      // Navigate to the admin policy privacy page with increased timeout
      await page.goto('http://127.0.0.1:8000/admin/policy/privacy', { timeout: 60000 });
      
      // Wait for login form to be visible
      await page.waitForSelector('input[name="username"]', { timeout: 30000 });
      
      // Check if we need to login by looking for login form elements
      const loginHeading = page.getByRole('heading', { name: 'Login' });
      const isLoginPage = await loginHeading.isVisible({ timeout: 5000 }).catch(() => false);
      
      if (isLoginPage) {
        console.log('Login required, attempting to login...');
        
        // Fill login form
        await page.getByRole('textbox', { name: 'Username' }).fill('admin');
        await page.getByRole('textbox', { name: 'Password' }).fill('admin');
        await page.getByRole('button', { name: 'Login' }).click();
        
        // Wait for response
        await page.waitForTimeout(2000);
        await page.waitForLoadState('networkidle', { timeout: 30000 });
        
        // Navigate to the privacy page after login
        await page.goto('http://127.0.0.1:8000/admin/policy/privacy', { timeout: 30000 });
        await page.waitForLoadState('networkidle', { timeout: 15000 });
      }
    } catch (error) {
      console.log('Navigation error, retrying...', error.message);
      // Check if page is still valid before retrying
      if (!page.isClosed()) {
        try {
          await page.goto('http://127.0.0.1:8000/admin/policy/privacy', { timeout: 30000 });
          await page.waitForLoadState('networkidle', { timeout: 15000 });
        } catch (retryError) {
          console.log('Retry failed, continuing with test...', retryError.message);
        }
      }
    }
  });

  test('should load admin policy privacy page successfully', async ({ page }) => {
    // Check if the page title contains expected text
    await expect(page).toHaveTitle(/Z\.com/);
    
    // Check if the URL is correct (allow for redirects)
    const currentUrl = page.url();
    expect(currentUrl).toMatch(/\/admin\/policy\/privacy|redirect.*policy.*privacy/);
    
    // Check if Save button is present
    await expect(page.getByRole('button', { name: 'Save' })).toBeVisible();
    
    // Check if form is present
    await expect(page.locator('#policy_frm')).toBeVisible();
  });

  test('should display form validation for required fields', async ({ page }) => {
    // Wait for the form to be fully loaded
    await page.waitForSelector('button:has-text("Save")', { timeout: 10000 });
    
    // Try to save without filling required fields
    await page.getByRole('button', { name: 'Save' }).click();
    
    // Wait a moment for validation to appear
    await page.waitForTimeout(2000);
    
    // Check if we're still on the same page (validation prevented submission)
    const currentUrl = page.url();
    expect(currentUrl).toContain('/admin/policy/privacy');
    
    // Check if form is still visible (not redirected away)
    await expect(page.locator('#policy_frm')).toBeVisible();
  });

  test('should fill and save English form data', async ({ page }) => {
    // Switch to English language first
    await page.selectOption('#mwz-select-lang', 'en');
    await page.waitForLoadState('networkidle');
    
    // Wait for English form fields to be visible
    await page.waitForSelector('#name_en', { timeout: 10000 });
    
    // Fill English form fields
    await page.locator('#name_en').fill('Test Privacy Policy');
    await page.locator('textarea[name="desc_en"]').fill('This is a test description for privacy policy');
    
    // Fill rich text editor
    const richTextFrame = page.locator('#detail_en_ifr').contentFrame();
    await richTextFrame?.getByLabel('Rich Text Area. Press ALT-0 for help.').fill('This is a test privacy policy content with detailed information about data collection and usage.');
    
    // Save the form
    await page.getByRole('button', { name: 'Save' }).click();
    
    // Wait for save to complete
    await page.waitForLoadState('networkidle');
    
    // Verify the content is saved (check if the rich text editor shows the content)
    const savedContent = await richTextFrame?.getByLabel('Rich Text Area. Press ALT-0 for help.').textContent();
    expect(savedContent).toContain('This is a test privacy policy content');
  });

  test('should switch to Thai language and fill form', async ({ page }) => {
    // Wait for language selector to be loaded
    await page.waitForSelector('select, [role="combobox"]', { timeout: 10000 });
    
    // Switch to Thai language
    await page.selectOption('#mwz-select-lang', 'th');
    
    // Wait for language switch
    await page.waitForLoadState('networkidle');
    
    // Verify language has switched by checking the select value
    const selectedValue = await page.locator('#mwz-select-lang').inputValue();
    expect(selectedValue).toBe('th');
    
    // Fill Thai form fields
    await page.locator('#name_th').fill('นโยบายความเป็นส่วนตัว');
    await page.locator('textarea[name="desc_th"]').fill('คำอธิบายนโยบายความเป็นส่วนตัว');
    
    // Fill Thai rich text editor
    const thaiRichTextFrame = page.locator('#detail_th_ifr').contentFrame();
    await thaiRichTextFrame?.getByLabel('Rich Text Area. Press ALT-0 for help.').fill('นี่คือเนื้อหานโยบายความเป็นส่วนตัวที่ทดสอบการทำงานของระบบ');
    
    // Save the form
    await page.getByRole('button', { name: 'Save' }).click();
    
    // Wait for save to complete
    await page.waitForLoadState('networkidle');
    
    // Verify the Thai content is saved
    const savedThaiContent = await thaiRichTextFrame?.getByLabel('Rich Text Area. Press ALT-0 for help.').textContent();
    expect(savedThaiContent).toContain('นี่คือเนื้อหานโยบายความเป็นส่วนตัว');
  });

  test('should test rich text editor functionality', async ({ page }) => {
    // Click on the rich text editor
    const richTextFrame = page.locator('#detail_en_ifr').contentFrame();
    await richTextFrame?.getByLabel('Rich Text Area. Press ALT-0 for help.').click();
    
    // Test basic text formatting
    await richTextFrame?.getByLabel('Rich Text Area. Press ALT-0 for help.').fill('Test content for formatting');
    
    // Test bold formatting
    await page.getByRole('button', { name: 'Bold' }).click();
    await richTextFrame?.getByLabel('Rich Text Area. Press ALT-0 for help.').fill('Bold text test');
    
    // Test italic formatting
    await page.getByRole('button', { name: 'Italic' }).click();
    await richTextFrame?.getByLabel('Rich Text Area. Press ALT-0 for help.').fill('Italic text test');
    
    // Test underline formatting
    await page.getByRole('button', { name: 'Underline' }).click();
    await richTextFrame?.getByLabel('Rich Text Area. Press ALT-0 for help.').fill('Underlined text test');
    
    // Test text alignment
    await page.getByRole('button', { name: 'Align center' }).click();
    await richTextFrame?.getByLabel('Rich Text Area. Press ALT-0 for help.').fill('Centered text test');
    
    // Test bullet list
    await page.getByRole('button', { name: 'Bullet list' }).click();
    await richTextFrame?.getByLabel('Rich Text Area. Press ALT-0 for help.').fill('• Bullet point test');
    
    // Test numbered list
    await page.getByRole('button', { name: 'Numbered list' }).click();
    await richTextFrame?.getByLabel('Rich Text Area. Press ALT-0 for help.').fill('1. Numbered point test');
  });

  test('should test form tabs (Information and SEO)', async ({ page }) => {
    // Test Information tab (should be active by default)
    await expect(page.getByRole('link', { name: 'Information' })).toBeVisible();
    
    // Test SEO tab
    await page.getByRole('link', { name: 'SEO' }).click();
    await page.waitForLoadState('networkidle');
    
    // Verify SEO tab is active
    await expect(page.getByRole('link', { name: 'SEO' })).toBeVisible();
    
    // Go back to Information tab
    await page.getByRole('link', { name: 'Information' }).click();
    await page.waitForLoadState('networkidle');
    
    // Verify Information tab is active again
    await expect(page.getByRole('link', { name: 'Information' })).toBeVisible();
  });

  test('should test language switching functionality', async ({ page }) => {
    // Test switching from English to Thai
    await page.selectOption('#mwz-select-lang', 'th');
    await page.waitForLoadState('networkidle');
    
    // Verify Thai language is selected
    const thaiValue = await page.locator('#mwz-select-lang').inputValue();
    expect(thaiValue).toBe('th');
    
    // Test switching back to English
    await page.selectOption('#mwz-select-lang', 'en');
    await page.waitForLoadState('networkidle');
    
    // Verify English language is selected
    const englishValue = await page.locator('#mwz-select-lang').inputValue();
    expect(englishValue).toBe('en');
  });

  test('should test form field validation', async ({ page }) => {
    // Test empty form submission
    await page.getByRole('button', { name: 'Save' }).click();
    
    // Wait a moment for validation to appear
    await page.waitForTimeout(2000);
    
    // Check if we're still on the same page (validation prevented submission)
    const currentUrl = page.url();
    expect(currentUrl).toContain('/admin/policy/privacy');
    
    // Test with only name filled
    await page.locator('#name_en').fill('Test Name');
    await page.getByRole('button', { name: 'Save' }).click();
    await page.waitForTimeout(2000);
    
    // Check if still on same page (validation should prevent submission)
    const urlAfterName = page.url();
    expect(urlAfterName).toContain('/admin/policy/privacy');
    
    // Test with name and description filled
    await page.locator('textarea[name="desc_en"]').fill('Test Description');
    await page.getByRole('button', { name: 'Save' }).click();
    await page.waitForTimeout(2000);
    
    // Check if still on same page (validation should prevent submission)
    const urlAfterDesc = page.url();
    expect(urlAfterDesc).toContain('/admin/policy/privacy');
    
    // Test with all fields filled
    const richTextFrame = page.locator('#detail_en_ifr').contentFrame();
    await richTextFrame?.getByLabel('Rich Text Area. Press ALT-0 for help.').fill('Test content');
    await page.getByRole('button', { name: 'Save' }).click();
    
    // Should successfully save when all fields are filled
    await page.waitForLoadState('networkidle');
    const finalUrl = page.url();
    expect(finalUrl).toContain('/admin/policy/privacy');
  });

  test('should test sidebar navigation', async ({ page }) => {
    // Test Policy submenu
    await expect(page.getByRole('link', { name: 'detail' })).toBeVisible();
    await expect(page.getByRole('link', { name: 'log' })).toBeVisible();
    
    // Test clicking on log link
    await page.getByRole('link', { name: 'log' }).click();
    await page.waitForLoadState('networkidle');
    
    // Verify navigation to logs page
    expect(page.url()).toBe('http://127.0.0.1:8000/admin/policy/logs');
    
    // Navigate back to privacy page
    await page.goto('http://127.0.0.1:8000/admin/policy/privacy');
    await page.waitForLoadState('networkidle');
    
    // Verify we're back on the privacy page
    expect(page.url()).toBe('http://127.0.0.1:8000/admin/policy/privacy');
  });

  test('should test responsive design elements', async ({ page }) => {
    // Test mobile viewport
    await page.setViewportSize({ width: 375, height: 667 });
    await page.waitForLoadState('networkidle');
    
    // Verify form elements are still visible
    await expect(page.getByRole('button', { name: 'Save' })).toBeVisible();
    await expect(page.locator('#policy_frm')).toBeVisible();
    
    // Test tablet viewport
    await page.setViewportSize({ width: 768, height: 1024 });
    await page.waitForLoadState('networkidle');
    
    // Verify form elements are still visible
    await expect(page.getByRole('button', { name: 'Save' })).toBeVisible();
    
    // Reset to desktop viewport
    await page.setViewportSize({ width: 1280, height: 720 });
    await page.waitForLoadState('networkidle');
  });

  test('should test form data persistence', async ({ page }) => {
    // Fill form with test data
    await page.locator('#name_en').fill('Persistent Test Policy');
    await page.locator('textarea[name="desc_en"]').fill('This data should persist after save');
    
    const richTextFrame = page.locator('#detail_en_ifr').contentFrame();
    await richTextFrame?.getByLabel('Rich Text Area. Press ALT-0 for help.').fill('This content should be saved and persist across page reloads.');
    
    // Save the form
    await page.getByRole('button', { name: 'Save' }).click();
    await page.waitForLoadState('networkidle');
    
    // Reload the page
    await page.reload();
    await page.waitForLoadState('networkidle');
    
    // Verify data persists
    await expect(page.locator('#name_en')).toHaveValue('Persistent Test Policy');
    await expect(page.locator('textarea[name="desc_en"]')).toHaveValue('This data should persist after save');
    
    const reloadedRichTextFrame = page.locator('#detail_en_ifr').contentFrame();
    const savedContent = await reloadedRichTextFrame?.getByLabel('Rich Text Area. Press ALT-0 for help.').textContent();
    expect(savedContent).toContain('This content should be saved and persist across page reloads.');
  });
});
