import { test, expect } from '@playwright/test';

// Test configuration
const BASE_URL = 'http://127.0.0.1:8000';
const ADMIN_LOGIN_URL = `${BASE_URL}/admin`;
const USER_LIST_URL = `${BASE_URL}/admin/user/list`;

// Test credentials
const ADMIN_CREDENTIALS = {
  username: 'admin',
  password: 'admin'
};

test.describe('Admin User Edit Tests', () => {
  test.beforeEach(async ({ page }) => {
    try {
      // Navigate to admin login page first with increased timeout
      await page.goto(ADMIN_LOGIN_URL, { timeout: 60000 });
      
      // Wait for login form to be visible
      await page.waitForSelector('input[name="username"]', { timeout: 30000 });
      
      // Login as admin
      await page.getByRole('textbox', { name: 'Username' }).fill(ADMIN_CREDENTIALS.username);
      await page.getByRole('textbox', { name: 'Password' }).fill(ADMIN_CREDENTIALS.password);
      await page.getByRole('button', { name: 'Login' }).click();
      
      // Wait for login to complete
      await page.waitForLoadState('networkidle', { timeout: 30000 });
      
      // Navigate to user list
      await page.goto(USER_LIST_URL, { timeout: 30000 });
      
      // Wait for the page to load completely
      await page.waitForLoadState('networkidle', { timeout: 30000 });
      
      // Wait for the table to be visible to ensure page is fully loaded
      await page.waitForSelector('table', { timeout: 15000 });
    } catch (error) {
      console.error('Setup failed:', error);
      throw error;
    }
  });

  test('should load admin user list page successfully', async ({ page }) => {
    // Verify page title
    await expect(page).toHaveTitle('Z.com');
    
    // Verify URL
    expect(page.url()).toBe(USER_LIST_URL);
    
    // Verify page elements are present
    await expect(page.getByRole('table')).toBeVisible();
    await expect(page.getByRole('searchbox')).toBeVisible();
    await expect(page.getByRole('link', { name: 'Add' })).toBeVisible();
    await expect(page.getByRole('button', { name: 'Filter' })).toBeVisible();
    
    // Verify table headers
    await expect(page.getByRole('cell', { name: 'Sequence' })).toBeVisible();
    await expect(page.getByRole('cell', { name: 'Name: activate to sort column' })).toBeVisible();
    await expect(page.getByRole('cell', { name: 'Username : activate to sort' })).toBeVisible();
    await expect(page.getByRole('cell', { name: 'Role' })).toBeVisible();
    await expect(page.getByRole('cell', { name: 'Updated' })).toBeVisible();
    await expect(page.getByRole('cell', { name: 'Manage' })).toBeVisible();
  });

  test('should display user data in table', async ({ page }) => {
    // Check if user data is displayed
    const userRows = page.getByRole('row').filter({ hasText: 'admindd' });
    await expect(userRows).toHaveCount(1);
    
    // Verify user data content
    await expect(page.getByRole('cell', { name: 'admindd' })).toBeVisible();
    await expect(page.getByRole('cell', { name: 'General Admin' })).toBeVisible();
    
    // Verify edit buttons are present using a more specific selector
    const editButton = page.locator('a[href*="/edit/"]').first();
    await expect(editButton).toBeVisible();
  });

  test('should navigate to edit page when clicking edit button', async ({ page }) => {
    // Click on the first edit button (for admindd user)
    const editButton = page.locator('a[href*="/edit/"]').first();
    await editButton.click();
    
    // Verify navigation to edit page
    await expect(page).toHaveURL(/.*\/admin\/user\/list\/edit\/\d+/);
    
    // Wait for edit form to load
    await page.waitForLoadState('networkidle');
  });

  test('should display edit form with all required fields', async ({ page }) => {
    // Navigate to edit page
    const editButton = page.locator('a[href*="/edit/"]').first();
    await editButton.click();
    
    // Wait for form to load
    await page.waitForLoadState('networkidle');
    
    // Verify form elements are present
    await expect(page.getByRole('textbox', { name: 'Please specify Name' })).toBeVisible();
    await expect(page.getByRole('textbox', { name: 'Please specify Username' })).toBeVisible();
    await expect(page.getByRole('textbox', { name: 'Please specify Email' })).toBeVisible();
    await expect(page.getByRole('textbox', { name: 'Please specify Password' })).toBeVisible();
    await expect(page.getByRole('textbox', { name: 'Please specify Confirm Password' })).toBeVisible();
    await expect(page.getByRole('combobox')).toBeVisible();
    await expect(page.getByRole('switch')).toBeVisible();
    
    // Verify form buttons
    await expect(page.getByRole('button', { name: 'Save' })).toBeVisible();
    await expect(page.getByRole('link', { name: 'Cancel' })).toBeVisible();
  });

  test('should populate form with existing user data', async ({ page }) => {
    // Navigate to edit page
    const editButton = page.locator('a[href*="/edit/"]').first();
    await editButton.click();
    
    // Wait for form to load
    await page.waitForLoadState('networkidle');
    
    // Verify form is populated with existing data
    const nameField = page.getByRole('textbox', { name: 'Please specify Name' });
    const usernameField = page.getByRole('textbox', { name: 'Please specify Username' });
    const emailField = page.getByRole('textbox', { name: 'Please specify Email' });
    
    await expect(nameField).toHaveValue('admindd');
    await expect(usernameField).toHaveValue('admindd');
    await expect(emailField).toHaveValue('admindd@gmail.com');
  });

  test('should allow editing user name and email', async ({ page }) => {
    // Navigate to edit page
    const editButton = page.locator('a[href*="/edit/"]').first();
    await editButton.click();
    
    // Wait for form to load
    await page.waitForLoadState('networkidle');
    
    // Edit name field
    const nameField = page.getByRole('textbox', { name: 'Please specify Name' });
    await nameField.clear();
    await nameField.fill('Admin Updated');
    
    // Edit email field
    const emailField = page.getByRole('textbox', { name: 'Please specify Email' });
    await emailField.clear();
    await emailField.fill('admin.updated@example.com');
    
    // Verify changes
    await expect(nameField).toHaveValue('Admin Updated');
    await expect(emailField).toHaveValue('admin.updated@example.com');
  });

  test('should handle role dropdown interaction', async ({ page }) => {
    // Navigate to edit page
    const editButton = page.locator('a[href*="/edit/"]').first();
    await editButton.click();
    
    // Wait for form to load
    await page.waitForLoadState('networkidle');
    
    // Click on role dropdown
    const roleDropdown = page.getByRole('combobox', { name: 'General Admin' });
    await roleDropdown.click();
    
    // Verify dropdown is expanded
    await expect(page.getByRole('listbox')).toBeVisible();
    await expect(page.getByRole('option', { name: 'General Admin' })).toBeVisible();
  });

  test('should toggle status switch', async ({ page }) => {
    // Navigate to edit page
    const editButton = page.locator('a[href*="/edit/"]').first();
    await editButton.click();
    
    // Wait for form to load
    await page.waitForLoadState('networkidle');
    
    // Get initial state of status switch
    const statusSwitch = page.getByRole('switch', { name: 'display API' });
    const initialChecked = await statusSwitch.isChecked();
    
    // Toggle the switch
    await statusSwitch.click();
    
    // Verify state changed
    const newChecked = await statusSwitch.isChecked();
    expect(newChecked).not.toBe(initialChecked);
  });

  test('should navigate back to list when clicking Cancel', async ({ page }) => {
    // Navigate to edit page
    const editButton = page.locator('a[href*="/edit/"]').first();
    await editButton.click();
    
    // Wait for form to load
    await page.waitForLoadState('networkidle');
    
    // Click Cancel button
    await page.getByRole('link', { name: 'Cancel' }).click();
    
    // Verify navigation back to list
    await expect(page).toHaveURL(USER_LIST_URL);
    await expect(page.getByRole('table')).toBeVisible();
  });

  test('should show password fields as masked', async ({ page }) => {
    // Navigate to edit page
    const editButton = page.locator('a[href*="/edit/"]').first();
    await editButton.click();
    
    // Wait for form to load
    await page.waitForLoadState('networkidle');
    
    // Verify password fields are masked
    const passwordField = page.getByRole('textbox', { name: 'Please specify Password' });
    const confirmPasswordField = page.getByRole('textbox', { name: 'Please specify Confirm Password' });
    
    await expect(passwordField).toHaveAttribute('type', 'password');
    await expect(confirmPasswordField).toHaveAttribute('type', 'password');
    
    // Verify password fields show masked values
    await expect(passwordField).toHaveValue('********');
    await expect(confirmPasswordField).toHaveValue('********');
  });

  test('should have avatar upload section', async ({ page }) => {
    // Navigate to edit page
    const editButton = page.locator('a[href*="/edit/"]').first();
    await editButton.click();
    
    // Wait for form to load
    await page.waitForLoadState('networkidle');
    
    // Verify avatar section is present
    await expect(page.getByText('Avatar')).toBeVisible();
    await expect(page.getByText('Click to upload')).toBeVisible();
    await expect(page.getByText('recommended size 100 x 100px')).toBeVisible();
  });

  test('should validate required fields', async ({ page }) => {
    // Navigate to edit page
    const editButton = page.locator('a[href*="/edit/"]').first();
    await editButton.click();
    
    // Wait for form to load
    await page.waitForLoadState('networkidle');
    
    // Clear required fields
    const nameField = page.getByRole('textbox', { name: 'Please specify Name' });
    const emailField = page.getByRole('textbox', { name: 'Please specify Email' });
    
    await nameField.clear();
    await emailField.clear();
    
    // Try to save (this should trigger validation)
    await page.getByRole('button', { name: 'Save' }).click();
    
    // Wait for validation messages to appear
    await page.waitForTimeout(1000);
    
    // Check for validation errors (adjust selectors based on actual implementation)
    // These are common validation error patterns
    const hasValidationErrors = await page.locator('.error, .invalid, [class*="error"], [class*="invalid"]').count() > 0;
    
    // If validation is working, we should either see errors or stay on the same page
    if (hasValidationErrors) {
      await expect(page.locator('.error, .invalid, [class*="error"], [class*="invalid"]').first()).toBeVisible();
    } else {
      // If no validation errors are shown, at least verify we're still on the edit page
      await expect(page).toHaveURL(/.*\/edit\/\d+/);
    }
  });
});
